package host.anzo.commons.concurrent;

import java.util.concurrent.locks.ReentrantLock;

/**
 * A CloseableReentrantLock is a specialized implementation of a ReentrantLock
 * that implements the AutoCloseable interface. This allows the lock to be used
 * in a try-with-resources statement, ensuring that the lock is properly released
 * when it is no longer needed.
 * <br><br>
 * This lock is fair, meaning that threads will acquire the lock in the order
 * they requested it.
 * @author tsa
 * @since 20.11.2016
 */
public class CloseableReentrantLock extends ReentrantLock implements AutoCloseable {

	/**
	 * Constructs a new CloseableReentrantLock with fairness set to true.
	 * This ensures that the lock favors threads that are waiting to acquire it.
	 */
	public CloseableReentrantLock() {
		super(true);
	}

	/**
	 * Acquires the lock, blocking until it is available.
	 *
	 * @return this instance of CloseableReentrantLock, allowing for method chaining.
	 */
	public CloseableReentrantLock open() {
		this.lock();
		return this;
	}

	/**
	 * Releases the lock. This method is called automatically when the
	 * try-with-resources statement is exited.
	 *
	 * @throws IllegalMonitorStateException if the current thread does not hold the lock.
	 */
	@Override
	public void close() {
		this.unlock();
	}
}
