package host.anzo.commons.datetime.dailytick;

import host.anzo.classindex.ClassIndex;
import host.anzo.commons.annotations.startup.StartupComponent;
import host.anzo.commons.enums.startup.EShutdownPriority;
import host.anzo.commons.utils.ClassUtils;
import host.anzo.core.service.ThreadPoolService;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import java.time.DayOfWeek;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author ANZO
 * @since 8/19/2021
 */
@Slf4j
@StartupComponent(value = "Service", shutdownPriority = EShutdownPriority.MAJOR)
public class DailyTickService {
    @Getter(lazy = true)
    private final static DailyTickService instance = new DailyTickService();

    private final List<Class<?>> consumerClasses = new ArrayList<>();

    private DailyTickService() {
        ClassIndex.getAnnotated(DailyTickable.class).forEach(item -> {
            final DailyTickable annotation = item.getAnnotation(DailyTickable.class);
            if (annotation != null) {
                if (IDailyTickable.class.isAssignableFrom(item)) {
                    consumerClasses.add(item);
                }
                else {
                    log.error("Found marked with DailyTickable annotation class without IDailyTickable implementation: {}", item.getSimpleName());
                }
            }
        });
        log.info("Found [{}] daily tickable classes.", consumerClasses.size());

        for (EDailyTickType dailyTickType : EDailyTickType.values()) {
            final long timeToNextTick = dailyTickType.getTimeToNextTick();
            ThreadPoolService.getInstance().scheduleAtFixedRate(() -> runDailyTask(dailyTickType), timeToNextTick, dailyTickType.getTickPeriod(), TimeUnit.MILLISECONDS, "DailyTickService.runDailyTask(" + dailyTickType + ")");
            log.info("Scheduled DailyTickType.{} call in [{}] minutes.", dailyTickType, TimeUnit.MINUTES.convert(timeToNextTick, TimeUnit.MILLISECONDS));
        }
    }

    /**
     * Runs registered daily task of specified type.
     * @param dailyTickType daily tick type
     */
    private void runDailyTask(EDailyTickType dailyTickType) {
        try {
            final DayOfWeek dayOfWeek = LocalDate.now().getDayOfWeek();
            ThreadPoolService.getInstance().submitForkJoinGet(() -> consumerClasses.parallelStream().forEach(clazz -> {
                final Object object = ClassUtils.singletonInstance(clazz);
                if (object != null) {
                    if (object instanceof IDailyTickable) {
                        ((IDailyTickable)object).onDailyTick(dayOfWeek, dailyTickType);
                    }
                }
            }), "DailyTickService.tickTask("+ dailyTickType.toString() + ")");
            log.info("Daily task with type EDailyTickType.{} successfully completed.", dailyTickType);
        }
        catch (Exception e) {
            log.error("Error while running daily task type=[{}]", dailyTickType, e);
        }
    }
}