package host.anzo.core.service;

import com.maxmind.geoip2.DatabaseReader;
import com.maxmind.geoip2.WebServiceClient;
import com.maxmind.geoip2.model.AbstractCityResponse;
import com.maxmind.geoip2.model.CityResponse;
import com.maxmind.geoip2.model.InsightsResponse;
import com.maxmind.geoip2.record.Traits;
import host.anzo.commons.annotations.startup.StartupComponent;
import host.anzo.commons.collection.StatsSet;
import host.anzo.commons.utils.StringUtilsEx;
import it.unimi.dsi.fastutil.objects.Object2ObjectLinkedOpenHashMap;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import java.io.IOException;
import java.net.InetAddress;
import java.net.URL;

/**
 * Provides GeoIP lookup services using MaxMind databases and web services.
 * <p>
 * This service initializes on startup and offers methods to retrieve geographical information
 * based on IP addresses. It supports both local database lookups (GeoLite2-City) and
 * web service-based insights (requires valid MaxMind credentials).
 *
 * <p><b>Configuration:</b>
 * <ul>
 *   <li>Requires {@code GeoLite2-City.mmdb} in classpath</li>
 *   <li>Web service requires system properties {@code maxMindAccountId} and {@code maxMindLicenseKey}</li>
 * </ul>
 *
 * @author ANZO
 * @since 10/24/2018
 */
@Slf4j
@StartupComponent("Service")
public class GeoIpService {
	@Getter(lazy = true)
	private static final GeoIpService instance = new GeoIpService();

	private WebServiceClient webServiceClient;
	private DatabaseReader databaseReader;

	private GeoIpService() {
		try {
			final URL databaseUrl = getClass().getClassLoader().getResource("GeoLite2-City.mmdb");
			if (databaseUrl != null) {
				databaseReader = new DatabaseReader.Builder(databaseUrl.openStream()).build();
				log.info("Loaded [{}] database (build date: [{}])",
						databaseReader.getMetadata().getDatabaseType(),
						databaseReader.getMetadata().getBuildDate()
				);
			}
			else {
				log.warn("Cannot find GeoLite2-City.mmdb in classpath.");
			}
			final int maxMindAccountId = Integer.parseInt(System.getProperty("maxMindAccountId", "0"));
			final String maxMindLicenseKey = System.getProperty("maxMindLicenseKey", "");
			if (maxMindAccountId != 0 && StringUtilsEx.isNoneEmpty(maxMindLicenseKey)) {
				webServiceClient = new WebServiceClient.Builder(
						Integer.parseInt(System.getProperty("maxMindAccountId", "0")),
						System.getProperty("maxMindLicenseKey", "")
				).build();
				log.info("Initialized MaxMind GeoIP web service client for accountId=[{}]", maxMindAccountId);
			}
		} catch (IOException e) {
			log.error("Error while loading Geo IP data", e);
		}
	}

	/**
	 * Retrieves city-level information for a given IP address using the local database.
	 *
	 * @param ipAddress IP address to query (e.g., "192.168.1.1")
	 * @return {@link CityResponse} with geographical data, or {@code null} if:
	 *         <ul>
	 *           <li>Database not initialized</li>
	 *           <li>Invalid IP format</li>
	 *           <li>Lookup error occurs</li>
	 *         </ul>
	 */
	public CityResponse getCity(String ipAddress) {
		if (databaseReader == null) {
			return null;
		}
		try {
			return databaseReader.city(InetAddress.getByName(ipAddress));
		} catch (Exception e) {
			return null;
		}
	}

	/**
	 * Retrieves the country name associated with an IP address using the local database.
	 *
	 * @param ipAddress IP address to query
	 * @return Country name as String, or {@code null} if:
	 *         <ul>
	 *           <li>Database not initialized</li>
	 *           <li>Country data unavailable</li>
	 *           <li>Lookup error occurs</li>
	 *         </ul>
	 */
	public String getCountryName(String ipAddress) {
		if (databaseReader == null) {
			return null;
		}
		try {
			return databaseReader.country(InetAddress.getByName(ipAddress)).getCountry().getName();
		} catch (Exception e) {
			return null;
		}
	}

	/**
	 * Retrieves detailed insights for an IP address using MaxMind's web service.
	 * <p>
	 * Requires valid {@code maxMindAccountId} and {@code maxMindLicenseKey} system properties.
	 *
	 * @param ipAddress IP address to query
	 * @return {@link InsightsResponse} with extended data, or {@code null} if:
	 *         <ul>
	 *           <li>Web service client not initialized</li>
	 *           <li>Invalid credentials</li>
	 *           <li>Lookup error occurs</li>
	 *         </ul>
	 */
	public InsightsResponse getInsights(String ipAddress) {
		if (webServiceClient == null) {
			return null;
		}
		try {
			return webServiceClient.insights(InetAddress.getByName(ipAddress));
		} catch (Exception e) {
			return null;
		}
	}

	/**
	 * Builds a {@link StatsSet} with geo-location data for an IP address.
	 * <p>
	 * Includes different data tiers based on the {@code full} parameter:
	 * <ul>
	 *   <li><b>Basic</b> ({@code full=false}): City, country, coordinates, network traits</li>
	 *   <li><b>Full</b> ({@code full=true}): ISP, organization, proxy detection, mobile codes</li>
	 * </ul>
	 *
	 * @param ipAddress IP address to analyze
	 * @param full      {@code true} to include extended insights data
	 * @return Populated {@link StatsSet}, or {@code null} on any error
	 * @see StatsSet Keys include:
	 *       <ul>
	 *         <li>City, Country, CountryCode, EuroUnion</li>
	 *         <li>AccuracyRadius, Latitude, Longitude, Population</li>
	 *         <li>Network, PublicProxy, AnonymousProxy, SatelliteProvider</li>
	 *         <li>Full-mode only: Domain, ISP, ConnectionType, UserType, etc.</li>
	 *       </ul>
	 */
	public StatsSet getInfo(String ipAddress, boolean full) {
		try {
			StatsSet set = new StatsSet(Object2ObjectLinkedOpenHashMap::new);
			AbstractCityResponse response = full ? getInsights(ipAddress) : getCity(ipAddress);

			set.putString("--- GeoIP Service ---", full ? "Full" : "Short");
			if (response != null) {
				// Basic fields
				set.putString("City", response.getCity().getName());
				set.putString("Country", response.getCountry().getName());
				set.putString("CountryCode", response.getCountry().getIsoCode());
				set.putBoolean("EuroUnion", response.getCountry().isInEuropeanUnion());
				set.putObject("AccuracyRadius", response.getLocation().getAccuracyRadius());
				set.putObject("Latitude", response.getLocation().getLatitude());
				set.putObject("Longitude", response.getLocation().getLongitude());
				set.putObject("Population", response.getLocation().getPopulationDensity());

				// Traits fields
				if (response.getTraits() instanceof Traits traits) {
					set.putObject("Network", traits.getNetwork());
					set.putBoolean("PublicProxy", traits.isPublicProxy());
					set.putBoolean("AnonymousProxy", traits.isAnonymousProxy());
					set.putBoolean("SatelliteProvider", traits.isSatelliteProvider());

					// Extended fields for full mode
					if (full) {
						set.putString("Domain", traits.getDomain());
						set.putString("ISP", traits.getIsp());
						set.putEnum("ConnectionType", traits.getConnectionType());
						set.putString("UserType", traits.getUserType());
						set.putObject("UserCount", traits.getUserCount());
						set.putString("Organization", traits.getOrganization());
						set.putObject("AutonomousSystemNumber", traits.getAutonomousSystemNumber());
						set.putString("AutonomousSystemOrganization", traits.getAutonomousSystemOrganization());
						set.putString("MobileCountryCode", traits.getMobileCountryCode());
						set.putString("MobileNetworkCode", traits.getMobileNetworkCode());
						set.putBoolean("Anonymous", traits.isAnonymous());
						set.putBoolean("AnonymousVpn", traits.isAnonymousVpn());
						set.putBoolean("HostingProvider", traits.isHostingProvider());
						set.putBoolean("ResidentialProxy", traits.isResidentialProxy());
						set.putBoolean("LegitimateProxy", traits.isLegitimateProxy());
						set.putBoolean("TorExitNode", traits.isTorExitNode());
						set.putObject("StaticIpScore", traits.getStaticIpScore());
					}
				}
			}
			return set;
		} catch (Exception ignored) {
		}
		return null;
	}
}