package host.anzo.commons.utils;

import org.jetbrains.annotations.NotNull;

import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.concurrent.TimeUnit;

/**
 * Utility class for handling date and time operations.
 * This class provides methods for converting between different date and time formats,
 * calculating time spans, and manipulating date and time values.
 *
 * @author ANZO
 * @since 9/20/2018
 */
public class DateTimeUtils {
	/**
	 * The maximum SQL date represented as a LocalDateTime.
	 */
	public static final LocalDateTime MAX_SQL_DATE = LocalDateTime.of(2099, 1, 1, 0, 0, 0);

	/**
	 * DateTimeFormatter for formatting date and time in the pattern "yyyy-MM-dd HH:mm:ss".
	 */
	private static final DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

	/**
	 * Converts a LocalDateTime to epoch milliseconds.
	 *
	 * @param localDateTime the LocalDateTime to convert
	 * @return the epoch milliseconds representation of the given LocalDateTime
	 */
	public static long toEpochMillis(@NotNull LocalDateTime localDateTime) {
		return localDateTime.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
	}

	/**
	 * Returns a string representation of the international time for the given epoch milliseconds.
	 *
	 * @param millis the epoch milliseconds to convert
	 * @return a formatted string containing the time in system, EDT, and CET time zones
	 */
	public static @NotNull String getInternationalTime(long millis) {
		return "[SYS: " + LocalDateTime.ofInstant(Instant.ofEpochMilli(millis), ZoneId.systemDefault()).format(dateFormatter) + "] / " +
				"[EDT: " + LocalDateTime.ofInstant(Instant.ofEpochMilli(millis), ZoneId.of("GMT-04:00")).format(dateFormatter) + "] / " +
				"[CET: " + LocalDateTime.ofInstant(Instant.ofEpochMilli(millis), ZoneId.of("GMT+02:00")).format(dateFormatter) + "]";
	}

	/**
	 * Returns a string representation of the international time for the given epoch milliseconds.
	 *
	 * @return a formatted string containing the time in system, EDT, and CET time zones
	 */
	public static @NotNull String getInternationalTime() {
		return getInternationalTime(System.currentTimeMillis());
	}

	/**
	 * Formats the given LocalDateTime into a string using the defined date format.
	 *
	 * @param localDateTime the LocalDateTime to format
	 * @return a formatted string representation of the given LocalDateTime
	 */
	public static @NotNull String getFormattedDateTime(@NotNull LocalDateTime localDateTime) {
		return localDateTime.format(dateFormatter);
	}

	/**
	 * Checks if two epoch dates fall on the same day.
	 *
	 * @param date1 the first date in epoch format
	 * @param date2 the second date in epoch format
	 * @return {@code true} if date1 is the same day as date2, {@code false} otherwise
	 */
	public static boolean isSameDay(long date1, long date2) {
		final LocalDateTime localDateTime1 = LocalDateTime.ofInstant(Instant.ofEpochMilli(date1), ZoneId.systemDefault());
		final LocalDateTime localDateTime2 = LocalDateTime.ofInstant(Instant.ofEpochMilli(date2), ZoneId.systemDefault());
		return localDateTime1.getDayOfYear() == localDateTime2.getDayOfYear();
	}

	/**
	 * Parses a time span string in the format "HH:mm:ss" and converts it to the specified time unit.
	 *
	 * @param timeSpan the time span string to parse
	 * @param unit the time unit to convert to
	 * @return the time span in the specified time unit, or -1 if the format is invalid
	 */
	public static int parseTimeSpan(@NotNull String timeSpan, TimeUnit unit) {
		final String[] timeSpanData = timeSpan.split(":");
		if (timeSpanData.length == 3) {
			int resultSeconds = Integer.parseInt(timeSpanData[0]) * 3600;
			resultSeconds += Integer.parseInt(timeSpanData[1]) * 60;
			resultSeconds += Integer.parseInt(timeSpanData[2]);
			return (int)unit.convert(resultSeconds, TimeUnit.SECONDS);
		}
		return -1;
	}

	/**
	 * Converts a number of seconds into a time span string in the format "HH:mm:ss".
	 *
	 * @param seconds the number of seconds to convert
	 * @return a formatted string representation of the time span
	 */
	public static String toTimeSpan(long seconds) {
		return String.format("%02d:%02d:%02d", seconds / 3600, (seconds % 3600) / 60, seconds % 60);
	}

	/**
	 * Parses an integer representing a day of the week and returns the corresponding DayOfWeek enum value.
	 *
	 * @param day the day of the week (SUNDAY=0, MONDAY=6)
	 * @return the corresponding DayOfWeek enum value
	 */
	public static DayOfWeek parseDay(int day) {
		if (day == 0) {
			return DayOfWeek.SUNDAY;
		}
		return DayOfWeek.values()[day - 1];
	}

	/**
	 * Gets the current LocalDateTime based on the system's current time.
	 *
	 * @return the current LocalDateTime
	 */
	public static LocalDateTime getLocalDateTime() {
		return LocalDateTime.ofInstant(Instant.ofEpochMilli(System.currentTimeMillis()), ZoneId.systemDefault());
	}

	/**
	 * Converts an epoch time to a LocalDateTime.
	 *
	 * @param epochTime the epoch time to convert
	 * @return the corresponding LocalDateTime
	 */
	public static LocalDateTime getLocalDateTime(long epochTime) {
		return LocalDateTime.ofInstant(Instant.ofEpochMilli(epochTime), ZoneId.systemDefault());
	}

	/**
	 * Converts an epoch time to a ZonedDateTime.
	 *
	 * @param epochTime the epoch time to convert
	 * @return the corresponding ZonedDateTime
	 */
	public static ZonedDateTime getZonedDateTime(long epochTime) {
		return ZonedDateTime.ofInstant(Instant.ofEpochMilli(epochTime), ZoneId.systemDefault());
	}

	/**
	 * Converts an epoch time to a LocalDate.
	 *
	 * @param epochTime the epoch time to convert
	 * @return the corresponding LocalDate
	 */
	public static LocalDate getLocalDate(long epochTime) {
		return LocalDateTime.ofInstant(Instant.ofEpochMilli(epochTime), ZoneId.systemDefault()).toLocalDate();
	}

	/**
	 * Calculates the time passed from the start of the current day in the specified time unit.
	 *
	 * @param timeInMillis the epoch time stamp
	 * @param timeUnit the time unit to convert to
	 * @return the time passed from the start of the current day in the specified time unit
	 */
	public static long getCurrentDayPassedTime(long timeInMillis, @NotNull TimeUnit timeUnit) {
		return timeUnit.convert(timeInMillis - getLocalDate(timeInMillis).atStartOfDay().atZone(ZoneId.systemDefault()).toInstant().toEpochMilli(), TimeUnit.MILLISECONDS);
	}

	/**
	 * Gets the epoch time for the specified hour of the current day.
	 *
	 * @param timeInMillis the epoch time stamp
	 * @param hour the specified hour
	 * @return the epoch time at the specified hour of the current day
	 */
	public static long getCurrentDayHourTime(long timeInMillis, int hour) {
		return getCurrentDayHourTime(timeInMillis, hour, 0);
	}

	/**
	 * Gets the epoch time for the specified hour & minute of the current day.
	 *
	 * @param hour the specified hour
	 * @param minute the specified minute
	 * @return the epoch time at the specified hour of the current day
	 */
	public static long getCurrentDayHourTime(long timeInMillis, int hour, int minute) {
		return getLocalDateTime(timeInMillis).withHour(0).withMinute(0).withSecond(0).plusHours(hour).plusMinutes(minute).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
	}

	/**
	 * Gets the epoch time for the start of the current month.
	 *
	 * @param timeInMillis the epoch time stamp
	 * @return the epoch time for the start of the current month
	 */
	public static long getCurrentMonthStartTime(long timeInMillis) {
		final LocalDate localDate = getLocalDate(timeInMillis);
		final LocalDate currentMonthStartDay = LocalDate.of(localDate.getYear(), localDate.getMonth(), 1);
		return currentMonthStartDay.atStartOfDay(ZoneId.systemDefault()).toInstant().toEpochMilli();
	}

	/**
	 * Gets the epoch time for the start of the current month.
	 *
	 * @return the epoch time for the start of the current month
	 */
	public static long getCurrentMonthStartTime() {
		return getCurrentMonthStartTime(System.currentTimeMillis());
	}

	/**
	 * Gets the epoch time for the next specified hour and minute.
	 *
	 * @param timeInMillis the epoch time stamp
	 * @param hour the specified hour
	 * @param minute the specified minute
	 * @return the epoch time for the next specified hour and minute
	 */
	public static long getNextHourTime(long timeInMillis, int hour, int minute) {
		return getNextHourTime(getLocalDateTime(timeInMillis), hour, minute).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
	}

	/**
	 * Gets the epoch time for the next specified hour and minute.
	 *
	 * @param hour the specified hour
	 * @param minute the specified minute
	 * @return the epoch time for the next specified hour and minute
	 */
	public static long getNextHourTime(int hour, int minute) {
		return getNextHourTime(System.currentTimeMillis(), hour, minute);
	}

	/**
	 * Gets the LocalDateTime for the next specified hour and minute.
	 *
	 * @param dateTime the current LocalDateTime
	 * @param hour the specified hour
	 * @param minute the specified minute
	 * @return the LocalDateTime for the next specified hour and minute
	 */
	public static LocalDateTime getNextHourTime(@NotNull LocalDateTime dateTime, int hour, int minute) {
		final LocalDateTime nextHour = dateTime.withHour(0).withMinute(0).withSecond(0)
				.plusHours(hour).plusMinutes(minute);
		return nextHour.isBefore(dateTime) ? getNextDayHourTime(dateTime, hour, minute) : nextHour;
	}

	/**
	 * Gets the epoch time for the next specified hour.
	 *
	 * @param timeInMillis the epoch time stamp
	 * @param hour the specified hour
	 * @return the epoch time for the next specified hour
	 */
	public static long getNextHourTime(long timeInMillis, int hour) {
		return getNextHourTime(timeInMillis, hour, 0);
	}

	/**
	 * Gets the epoch time for the next hour from the current time.
	 *
	 * @param timeInMillis the epoch time stamp
	 * @return the epoch time for the next hour from the current time
	 */
	public static long getNextHourTime(long timeInMillis) {
		final LocalDateTime localDateTime = getLocalDateTime(timeInMillis);
		return getNextHourTime(timeInMillis, localDateTime.getHour() + 1, 0);
	}

	/**
	 * Gets the epoch time for the next hour from the current time.
	 *
	 * @return the epoch time for the next hour from the current time
	 */
	public static long getNextHourTime() {
		return getNextHourTime(System.currentTimeMillis());
	}

	/**
	 * Gets the epoch time for the start of the next specified day of the week.
	 *
	 * @param timeInMillis the epoch time stamp
	 * @param dayOfWeek the specified day of the week
	 * @return the epoch time for the start of the next specified day of the week
	 */
	public static long getNextDayStartTime(long timeInMillis, DayOfWeek dayOfWeek) {
		return getLocalDateTime(timeInMillis).with(TemporalAdjusters.next(dayOfWeek)).withHour(0).withMinute(0).withSecond(0).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
	}

	/**
	 * Gets the epoch time for the start of the next day.
	 *
	 * @param timeInMillis the epoch time stamp
	 * @return the epoch time for the start of the next day
	 */
	public static long getNextDayStartTime(long timeInMillis) {
		return getNextDayHourTime(timeInMillis, 0, 0);
	}

	/**
	 * Gets the epoch time for the start of the next day.
	 *
	 * @return the epoch time for the start of the next day
	 */
	public static long getNextDayStartTime() {
		return getNextDayStartTime(System.currentTimeMillis());
	}

	/**
	 * Gets the epoch time for a specified hour the next day.
	 *
	 * @param timeInMillis the epoch time stamp
	 * @param hour the specified hour
	 * @return the epoch time for the specified hour the next day
	 */
	public static long getNextDayHourTime(long timeInMillis, int hour) {
		return getLocalDateTime(timeInMillis).withHour(0).withMinute(0).withSecond(0).plusDays(1).plusHours(hour).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
	}

	/**
	 * Gets the epoch time for a specified hour the next day.
	 *
	 * @param hour the specified hour
	 * @return the epoch time for the specified hour the next day
	 */
	public static long getNextDayHourTime(int hour) {
		return getNextDayHourTime(System.currentTimeMillis(), hour);
	}

	/**
	 * Gets the epoch time for a specified hour and minute the next day.
	 *
	 * @param timeInMillis the epoch time stamp
	 * @param hour the specified hour
	 * @param minute the specified minute
	 * @return the epoch time for the specified hour and minute the next day
	 */
	public static long getNextDayHourTime(long timeInMillis, int hour, int minute) {
		return getLocalDateTime(timeInMillis).withHour(0).withMinute(0).withSecond(0).plusDays(1).plusHours(hour).plusMinutes(minute).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
	}

	/**
	 * Gets the epoch time for a specified hour and minute the next day.
	 *
	 * @param hour the specified hour
	 * @param minute the specified minute
	 * @return the epoch time for the specified hour and minute the next day
	 */
	public static long getNextDayHourTime(int hour, int minute) {
		return getNextDayHourTime(System.currentTimeMillis(), hour, minute);
	}

	/**
	 * Gets the LocalDateTime for a specified hour and minute the next day.
	 *
	 * @param dateTime the LocalDateTime to calculate from
	 * @param hour the specified hour
	 * @param minute the specified minute
	 * @return the LocalDateTime for the specified hour and minute the next day
	 */
	public static @NotNull LocalDateTime getNextDayHourTime(@NotNull LocalDateTime dateTime, int hour, int minute) {
		return dateTime.withHour(0).withMinute(0).withSecond(0).plusDays(1).plusHours(hour).plusMinutes(minute);
	}

	/**
	 * Gets the epoch time for the start of the next month.
	 *
	 * @param timeInMillis the epoch time stamp
	 * @return the epoch time for the start of the next month
	 */
	public static long getNextMonthStartTime(long timeInMillis) {
		final LocalDate localDate = getLocalDate(timeInMillis);
		final LocalDate nextMonthDay = LocalDate.of(localDate.getYear(), localDate.getMonth(), 1).plusMonths(1);
		return nextMonthDay.atStartOfDay(ZoneId.systemDefault()).toInstant().toEpochMilli();
	}

	/**
	 * Gets the epoch time for the start of the next month.
	 *
	 * @return the epoch time for the start of the next month
	 */
	public static long getNextMonthStartTime() {
		return getNextMonthStartTime(System.currentTimeMillis());
	}

	/**
	 * Gets the time in milliseconds to the next specified day of the week start.
	 *
	 * @param timeInMillis the epoch time stamp
	 * @param dayOfWeek the specified day of the week
	 * @return the time in milliseconds to the next specified day of the week start
	 */
	public static long getTimeToNextDayStart(long timeInMillis, DayOfWeek dayOfWeek) {
		return Math.max(0, getNextDayStartTime(timeInMillis, dayOfWeek) - getLocalDateTime(timeInMillis).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli());
	}

	/**
	 * Gets the time in milliseconds to the next specified day of the week start.
	 *
	 * @param dayOfWeek the specified day of the week
	 * @return the time in milliseconds to the next specified day of the week start
	 */
	public static long getTimeToNextDayStart(DayOfWeek dayOfWeek) {
		return getTimeToNextDayStart(System.currentTimeMillis(), dayOfWeek);
	}

	/**
	 * Gets the time in milliseconds to the next day start.
	 *
	 * @param timeInMillis the epoch time stamp
	 * @return the time in milliseconds to the next day start
	 */
	public static long getTimeToNextDayStart(long timeInMillis) {
		return Math.max(0, getNextDayStartTime(timeInMillis) - getLocalDateTime(timeInMillis).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli());
	}

	/**
	 * Gets the time in milliseconds to the next day start.
	 *
	 * @return the time in milliseconds to the next day start
	 */
	public static long getTimeToNextDayStart() {
		return getTimeToNextDayStart(System.currentTimeMillis());
	}

	/**
	 * Gets the time in milliseconds to a specified date time in the specified time unit.
	 *
	 * @param dateTime the date time to calculate time to
	 * @param timeUnit the time unit to convert to
	 * @return the time to the specified date time in the specified time unit
	 */
	public static long getTimeTo(@NotNull LocalDateTime dateTime, @NotNull TimeUnit timeUnit) {
		return timeUnit.convert(Math.max(0, dateTime.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli() - LocalDateTime.now().atZone(ZoneId.systemDefault()).toInstant().toEpochMilli()), TimeUnit.MILLISECONDS);
	}

	/**
	 * Gets the time in milliseconds to the next specified hour and minute.
	 *
	 * @param hour the specified hour
	 * @param minute the specified minute
	 * @param timeUnit the time unit to convert to
	 * @return the time to the next specified hour and minute in the specified time unit
	 */
	public static long getTimeToNextHour(int hour, int minute, @NotNull TimeUnit timeUnit) {
		return timeUnit.convert(Math.max(0, DateTimeUtils.getNextHourTime(LocalDateTime.now(), hour, minute).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli() - LocalDateTime.now().atZone(ZoneId.systemDefault()).toInstant().toEpochMilli()), TimeUnit.MILLISECONDS);
	}
}