package host.anzo.commons.utils;

import org.jetbrains.annotations.NotNull;

import java.nio.ByteBuffer;

/**
 * @author ANZO
 */
public class HexUtils {
	/**
	 * Converts a string of space-separated hexadecimal values into a byte array.
	 *
	 * @param  string  the string containing space-separated hexadecimal values
	 * @return         the byte array converted from the string
	 */
	public static byte @NotNull [] stringToHexByteArray(@NotNull String string) {
		final String[] stringArray = string.split(" ");
		final byte[] bytes = new byte[stringArray.length];
		for (int i = 0; i < bytes.length; i++) {
			bytes[i] = (byte) Integer.parseInt(stringArray[i], 16);
		}
		return bytes;
	}

	/**
	 * Converts a string to a hexadecimal string representation.
	 *
	 * @param  string  the string to convert
	 * @return         the hexadecimal string representation of the input string
	 */
	public static @NotNull String stringToHexString(@NotNull String string) {
		final byte[] chars = string.getBytes();
		return byteArrayToHex(chars);
	}

	/**
	 * Converts a byte array to a hexadecimal string representation.
	 *
	 * @param  bytes  the byte array to convert
	 * @return        the hexadecimal string representation of the byte array
	 */
	public static @NotNull String byteArrayToHex(byte @NotNull [] bytes) {
		final StringBuilder out = new StringBuilder();
		for (byte c : bytes) {
			out.append(Integer.toHexString(c & 0xFF)).append(" ");
		}
		return out.toString();
	}

	/**
	 * Converts a hexadecimal string to a byte array.
	 *
	 * @param  str  the hexadecimal string to convert
	 * @return      the byte array converted from the string
	 */
	public static byte @NotNull [] hexStringToByteArray(@NotNull String str) {
		byte[] bytes = new byte[str.length() / 2];
		for (int i = 0; i < bytes.length; i++) {
			bytes[i] = (byte) Integer.parseInt(str.substring(2 * i, 2 * i + 2), 16);
		}
		return bytes;
	}

	/**
	 * Converts a byte array to a formatted hexadecimal table string.
	 *
	 * @param  data  the byte array containing the data to be converted
	 * @param  size  the number of bytes to convert
	 * @return       the formatted hexadecimal table string
	 */
	public static @NotNull String byteArrayToHexTable(byte[] data, int size) {
		final StringBuilder result = new StringBuilder();
		char[] ascii = new char[17];
		int i, j;
		ascii[16] = '\0';
		for (i = 0; i < size; ++i) {
			byte dataByte = data[i];
			result.append(String.format("%02X ", dataByte));
			if ((char)data[i] >= ' ' && (char)dataByte <= '~') {
				ascii[i % 16] = (char)dataByte;
			} else {
				ascii[i % 16] = '.';
			}
			if ((i+1) % 8 == 0 || i + 1 == size) {
				result.append(" ");
				if ((i+1) % 16 == 0) {
					result.append("|  ").append(ascii).append("\r\n");
				} else if (i+1 == size) {
					ascii[(i+1) % 16] = '\0';
					if ((i+1) % 16 <= 8) {
						result.append(" ");
					}
					for (j = (i+1) % 16; j < 16; ++j) {
						result.append("   ");
					}
					result.append("|  ").append(ascii).append("\r\n");
				}
			}
		}
		return result.toString();
	}

	/**
	 * Converts a byte array to a formatted hexadecimal table string.
	 *
	 * @param  data  the byte buffer containing the data to be converted
	 * @param  size  the number of bytes to convert
	 * @return       the formatted hexadecimal table string
	 */
	public static @NotNull String byteArrayToHexTable(@NotNull ByteBuffer data, int size) {
		return byteArrayToHexTable(data.array(), size);
	}

	/**
	 * Converts a byte array to a formatted hexadecimal table string.
	 *
	 * @param  data  the byte array to convert
	 * @return       the formatted hexadecimal table string
	 */
	public static @NotNull String byteArrayToHexTable(byte[] data) {
		return byteArrayToHexTable(ByteBuffer.wrap(data), data.length);
	}
}