package host.anzo.core.service;

import host.anzo.commons.annotations.startup.StartupComponent;
import host.anzo.commons.enums.startup.EShutdownPriority;
import host.anzo.commons.interfaces.startup.IShutdownable;
import host.anzo.commons.objectpool.IObjectPool;
import host.anzo.commons.objectpool.IPoolObject;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.jctools.queues.MpmcArrayQueue;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;

/**
 * @author Aristo
 * @since 3/15/2025
 */
@Slf4j
@StartupComponent(value = "Service", shutdownPriority = EShutdownPriority.MINOR)
public class ObjectPoolService implements IShutdownable {
	private static final @Getter(lazy = true) ObjectPoolService instance = new ObjectPoolService();

	private final @Getter Map<String, IObjectPool<?>> pools = new ConcurrentHashMap<>();

	private ObjectPoolService() {
	}

	public <T> IObjectPool<T> createPool(String poolName, Supplier<T> supplier, int[] params) {
		IObjectPool<T> pool = new ObjectPool<>(poolName, supplier, params);
		pools.put(poolName, pool);
		return pool;
	}

	@Override
	public void onShutdown() {
		for (IObjectPool<?> pool : pools.values()) {
			if (pool != null) {
				pool.onShutdown();
			}
		}
		pools.clear();
	}

	@Slf4j
	private static class ObjectPool<T> implements IObjectPool<T> {
		private final AtomicBoolean shutdown = new AtomicBoolean(false);
		private final String poolName;

		private final MpmcArrayQueue<T> internalPool;
		private final Supplier<T> supplier;

		private final AtomicLong created = new AtomicLong(0);
		private final AtomicLong destroyed = new AtomicLong(0);
		private final AtomicLong borrowed = new AtomicLong(0);
		private final AtomicLong returned = new AtomicLong(0);
		private final AtomicLong cleared = new AtomicLong(0);
		private final AtomicLong empty = new AtomicLong(0);
		private final AtomicLong errored = new AtomicLong(0);

		private ObjectPool(String poolName, Supplier<T> supplier, int @NotNull [] params) {
			this.poolName = poolName;

			this.internalPool = new MpmcArrayQueue<>(params[0]);
			this.supplier = supplier;

			for (int i = 0; i < params[1]; i++) {
				internalPool.offer(supplier.get());
				created.incrementAndGet();
			}

			log.info("Object pool {} created.", poolName);
		}

		@Override
		public @Nullable T borrowObject() {
			if (shutdown.get()) {
				return null;
			}
			try {
				T object = internalPool.relaxedPoll();
				if (object == null) {
					object = supplier.get();
					created.incrementAndGet();
				}
				borrowed.incrementAndGet();
				return object;
			} catch (Exception e) {
				log.error("Error borrowing object from pool {}.", poolName, e);
				errored.incrementAndGet();
				return null;
			}
		}

		@Override
		public void returnObject(T object) {
			if (shutdown.get() || object == null) {
				return;
			}
			try {
				if (object instanceof IPoolObject poolObject) {
					poolObject.beforeReturn();
					cleared.incrementAndGet();
				}
				if (internalPool.relaxedOffer(object)) {
					returned.incrementAndGet();
				} else {
					if (object instanceof IPoolObject poolObject) {
						poolObject.onDestroy();
						destroyed.incrementAndGet();
					}
				}
			} catch (Exception e) {
				log.warn("Error returning object to pool {}.", poolName, e);
				errored.incrementAndGet();
			}
		}

		@Override
		public void onShutdown() {
			if (shutdown.compareAndSet(false, true)) {
				while (!internalPool.isEmpty()) {
					T object = internalPool.poll();
					if (object instanceof IPoolObject poolObject) {
						poolObject.onDestroy();
						destroyed.incrementAndGet();
					}
				}
			}
		}

		@Override
		public String getName() {
			return poolName;
		}

		@Override
		public @NotNull String toString() {
			return "ObjectPool{" +
					poolName +
					",created=" + created.get() +
					",destroyed=" + destroyed.get() +
					",borrowed=" + borrowed.get() +
					",returned=" + returned.get() +
					",cleared=" + cleared.get() +
					",empty=" + empty.get() +
					",errored=" + errored.get() +
					'}';
		}
	}
}