package host.anzo.commons.emergency.memory.watchers;

import host.anzo.commons.emergency.memory.MemoryLeakDetector;
import host.anzo.core.config.EmergencyConfig;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;

import javax.management.Notification;
import javax.management.NotificationListener;
import java.lang.management.MemoryNotificationInfo;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.MemoryUsage;

/**
 * The PoolListener class implements the NotificationListener interface and is responsible
 * for handling memory usage notifications for a specific MemoryPoolMXBean. It listens for
 * threshold exceeded notifications and logs critical memory usage information.
 * @author ANZO
 */
@Slf4j(topic = "Memory")
public class PoolListener implements NotificationListener {
	private final MemoryPoolMXBean pool;
	
	PoolListener(MemoryPoolMXBean pool) {
		this.pool = pool;
	}

	/**
	 * Handles notifications related to memory usage thresholds. It checks if the notification
	 * type indicates that a memory threshold has been exceeded. If so, it logs the current
	 * memory usage percentage and triggers the MemoryLeakDetector if the usage exceeds a
	 * predefined limit.
	 *
	 * @param notification the notification received
	 * @param handback an optional context object
	 */
	@Override
	public void handleNotification(@NotNull Notification notification, Object handback) {
		if(!notification.getType().equals(MemoryNotificationInfo.MEMORY_THRESHOLD_EXCEEDED)
				&& !notification.getType().equals(MemoryNotificationInfo.MEMORY_COLLECTION_THRESHOLD_EXCEEDED))
			return;
		
		MemoryUsage usage = pool.getUsage();
		long used = usage.getUsed();
		long max = usage.getMax();
		long percent = Math.round((used / (double)max) * 100);
		
		log.warn("Critical memory usage {}% for {} pool. Usage: {}/{} mb.", percent, pool.getName(), used / 1024 / 1024, max / 1024 / 1024);

		if (percent > EmergencyConfig.MEMORY_LEAK_DETECT_PERCENT) {
			MemoryLeakDetector.getInstance().onMemoryLeakDetected();
		}
	}
}