package host.anzo.commons.threading;

import de.mxro.metrics.jre.Metrics;
import delight.async.properties.PropertyNode;
import host.anzo.commons.emergency.metric.Metric;
import host.anzo.commons.emergency.metric.MetricGroupType;
import host.anzo.commons.emergency.metric.MetricResult;
import host.anzo.core.config.EmergencyConfig;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;

import java.lang.reflect.UndeclaredThrowableException;

/**
 * A wrapper for {@link Runnable} tasks that provides error handling and metrics collection.
 */
@Slf4j
@Metric
@SuppressWarnings("unused")
public class RunnableWrapper implements Runnable {
	private static final PropertyNode RUNNABLE_WRAPPER_METRICS = Metrics.create();
	private static final PropertyNode RUNNABLE_WRAPPER_METRICS_VIRTUAL = Metrics.create();

	private final Runnable _r;
	private final boolean _v;
	private final String _n;

	/**
	 * Creates a new RunnableWrapper for the given Runnable task.
	 *
	 * @param r the Runnable task to wrap
	 */
	public RunnableWrapper(final Runnable r) {
		this(r, false);
	}

	/**
	 * Creates a new RunnableWrapper for the given Runnable task.
	 *
	 * @param r       the Runnable task to wrap
	 * @param virtual whether the task should be treated as virtual for metrics collection
	 */
	public RunnableWrapper(final Runnable r, final boolean virtual) {
		this(r, virtual, r.getClass().getSimpleName());
	}

	/**
	 * Creates a new RunnableWrapper for the given Runnable task.
	 *
	 * @param r    the Runnable task to wrap
	 * @param name the name to use for metrics collection
	 */
	public RunnableWrapper(final Runnable r, final String name) {
		this(r, false, name);
	}

	/**
	 * Creates a new RunnableWrapper for the given Runnable task.
	 *
	 * @param r       the Runnable task to wrap
	 * @param virtual whether the task should be treated as virtual for metrics collection
	 * @param name    the name to use for metrics collection
	 */
	public RunnableWrapper(final Runnable r, final boolean virtual, final String name) {
		_r = r;
		_v = virtual;
		_n = name != null ? name : r.getClass().getSimpleName();
	}

	/**
	 * Executes the wrapped Runnable task.
	 * Collects metrics on execution time and errors if metrics are enabled.
	 * If an exception occurs during the execution of the wrapped Runnable, it is logged and wrapped in an {@link UndeclaredThrowableException} before being rethrown.
	 *
	 * @throws UndeclaredThrowableException if an exception occurs during execution of the wrapped Runnable.
	 *                                      The original exception is wrapped within the {@link UndeclaredThrowableException} and can be retrieved using {@link UndeclaredThrowableException#getCause()}.
	 */
	@Override
	public void run() throws UndeclaredThrowableException {
		if (_r != null) {
			try {
				if (EmergencyConfig.ENABLE_METRICS) {
					final long start = System.nanoTime();
					_r.run();
					recordMetric((_v ? RUNNABLE_WRAPPER_METRICS_VIRTUAL : RUNNABLE_WRAPPER_METRICS), _n + "_time_ns", System.nanoTime() - start);
				} else {
					_r.run();
				}
			} catch (final Throwable e) {
				recordMetric((_v ? RUNNABLE_WRAPPER_METRICS_VIRTUAL : RUNNABLE_WRAPPER_METRICS), _n + "_error");
				log.error("Error while running RunnableWrapper:", e);
				throw new UndeclaredThrowableException(e);
			}
		}
	}

	/**
	 * Returns the metrics for non-virtual RunnableWrappers.
	 *
	 * @return the metrics for non-virtual RunnableWrappers
	 */
	public static @NotNull MetricResult getMetric() {
		final MetricResult rw = new MetricResult();
		rw.setMetricGroupType(MetricGroupType.THREADPOOL);
		rw.setName("RunnableWrapper");
		rw.setData(RUNNABLE_WRAPPER_METRICS.render().get());
		return rw;
	}

	/**
	 * Returns the metrics for virtual RunnableWrappers.
	 *
	 * @return the metrics for virtual RunnableWrappers
	 */
	public static MetricResult getMetricVirtual() {
		final MetricResult rwv = new MetricResult();
		rwv.setMetricGroupType(MetricGroupType.THREADPOOL);
		rwv.setName("RunnableWrapperVirtual");
		rwv.setData(RUNNABLE_WRAPPER_METRICS_VIRTUAL.render().get());
		return rwv;
	}

	private void recordMetric(PropertyNode metrics, String metricName, long value) {
		if (EmergencyConfig.ENABLE_METRICS) {
			metrics.record(Metrics.value(metricName, value));
		}
	}

	private void recordMetric(PropertyNode metrics, String metricName) {
		if (EmergencyConfig.ENABLE_METRICS) {
			metrics.record(Metrics.happened(metricName));
		}
	}
}