package host.anzo.commons.utils;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.*;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import spark.Request;
import spark.Response;

import java.io.IOException;

/**
 * Utility class for handling JSON serialization and deserialization using Jackson.
 * This class provides methods to convert JSON data to Java objects and vice versa.
 * @author ANZO
 * @since 08.06.2017
 */
@Slf4j
public class JsonUtils {
    /**
     * ObjectMapper instance for JSON processing.
     * Configured to handle various serialization features.
     */
    private static final ObjectMapper jsonMapper = new ObjectMapper();
    static {
        jsonMapper.setVisibility(jsonMapper.getSerializationConfig()
                .with(SerializationFeature.INDENT_OUTPUT)
                .with(SerializationFeature.WRITE_ENUMS_USING_INDEX)
                .with(MapperFeature.ACCEPT_CASE_INSENSITIVE_VALUES)
                .with(MapperFeature.ACCEPT_CASE_INSENSITIVE_ENUMS)
                .getDefaultVisibilityChecker()
                .withFieldVisibility(JsonAutoDetect.Visibility.ANY)
                .withGetterVisibility(JsonAutoDetect.Visibility.NONE)
                .withSetterVisibility(JsonAutoDetect.Visibility.NONE)
                .withCreatorVisibility(JsonAutoDetect.Visibility.NONE));
    }

    /**
     * Converts the body of a Spark Request to a Java object of the specified class.
     *
     * @param request the Spark Request containing the JSON body
     * @param clazz the class of the object to convert to
     * @param <T> the type of the object
     * @return the converted object
     * @throws JsonProcessingException if the JSON processing fails
     */
    public static <T> T jsonToObject(@NotNull Request request, Class<T> clazz) throws JsonProcessingException {
        return jsonMapper.readValue(request.body(), clazz);
    }

    /**
     * Converts a JSON string to a Java object of the specified class.
     *
     * @param body the JSON string to convert
     * @param clazz the class of the object to convert to
     * @param <T> the type of the object
     * @return the converted object
     * @throws JsonProcessingException if the JSON processing fails
     */
    public static <T> T jsonToObject(String body, Class<T> clazz) throws JsonProcessingException {
        return jsonMapper.readValue(body, clazz);
    }

    /**
     * Converts a Java object to a JSON string and sets the response type to application/json.
     *
     * @param response the Spark Response to set the content type
     * @param data the object to convert to JSON
     * @return the JSON string representation of the object
     */
    public static String dataToJson(@NotNull Response response, Object data) {
        response.type("application/json");
        try {
            return jsonMapper.writeValueAsString(data);
        }
        catch (JsonProcessingException e) {
            log.error("Error while mapping object to JSON", e);
        }
        return "";
    }

    /**
     * Converts a Java object to a JSON string.
     *
     * @param data the object to convert to JSON
     * @return the JSON string representation of the object
     */
    public static String dataToJson(Object data) {
        try {
            return jsonMapper.writeValueAsString(data);
        }
        catch (JsonProcessingException e){
            log.error("Error while mapping object to JSON", e);
        }
        return "";
    }

    /**
     * Custom deserializer for Boolean values that can handle various string representations and null values.
     * If the input cannot be parsed, an exception is thrown.
     */
    @SuppressWarnings("unused")
    public static class CustomBooleanDeserializer extends JsonDeserializer<Boolean> {
        @Override
        public Boolean deserialize(@NotNull JsonParser jp, DeserializationContext ctxt) throws IOException {
            final JsonToken currentToken = jp.getCurrentToken();

            if (currentToken.equals(JsonToken.VALUE_STRING)) {
                final String text = jp.getText().toLowerCase();
                if (text.equals("true") || text.equals("y") || text.equals("yes") || text.equals("1")) {
                    return Boolean.TRUE;
                }
                return Boolean.FALSE;
            }
            else if (currentToken.equals(JsonToken.VALUE_NULL)) {
                return Boolean.FALSE;
            }
            ctxt.reportInputMismatch(this, "Can't parse boolean value: " + jp.getText());
            return false;
        }
    }
}