package host.anzo.commons.collection;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * A wrapper over Map that allows you to safely work with different data types.
 * Supports automatic type conversion when retrieving values, as well as
 * adding values, primitive types, and Enums.
 * Also implements standard Map methods.
 * <br>
 * <b>IMPORTANT:</b> Even when using a thread-safe internal map, data consistency is not guaranteed when using custom methods.
 * All manipulations are performed with the value retrieved at the time of access, and if another thread changes the data at this time, this will not be taken into account.
 * <br>
 * If standard Map methods are used, then thread safety will directly depend on the type of the internal map.
 *
 *
 * @author Aristo
 */
@Slf4j
public class StatsSet implements Map<String, Object> {
	private final @NotNull Map<String, Object> internalMap;

	/**
	 * Creates a new {@link StatsSet} instance using the factory to create the internal map.
	 *
	 * @param mapFactory factory to create the internal map
	 */
	public StatsSet(@NotNull Supplier<Map<String, Object>> mapFactory) {
		this(mapFactory.get());
	}

	/**
	 * Creates a new {@link StatsSet} instance with the given internal map.
	 *
	 * @param map internal map
	 */
	private StatsSet(@NotNull Map<String, Object> map) {
		internalMap = map;
	}

	/**
	 * Creates a new {@link StatsSet} instance and wraps the given map.
	 *
	 * @param map the map to wrap
	 * @return a new {@link StatsSet} instance
	 */
	public static StatsSet wrap(@NotNull Map<String, Object> map) {
		return new StatsSet(map);
	}

	/**
	 * Parses the parameter string by the given separators and creates a {@link StatsSet}.
	 * For example: (key1)(spliter2)(value1)(spliter1)(key2)(spliter2)(value2)(spliter1)(keyN)(spliter2)(valueN)
	 *
	 * @param params parameter string
	 * @param splitter1 separator between key-value pairs
	 * @param splitter2 separator between key and value
	 * @param mapFactory factory for creating the internal map
	 * @return a new {@link StatsSet} instance with the data from the parameter string
	 */
	public static StatsSet parseParams(@Nullable String params, @NotNull String splitter1, @NotNull String splitter2, @NotNull Supplier<Map<String, Object>> mapFactory) {
		final Map<String, Object> map = mapFactory.get();
		if (params != null) {
			try {
				final String[] args = params.split(splitter1);

				for (String split : args) {
					map.put(split.split(splitter2)[0], split.split(splitter2)[1]);
				}
			} catch (Exception e) {
				log.error("Can't parse params: {}", params, e);
			}
		}
		return StatsSet.wrap(map);
	}

	/**
	 * Returns a boolean value by key.
	 *
	 * @param key key
	 * @return boolean value
	 * @throws IllegalArgumentException if value is not found or cannot be converted to boolean
	 */
	public boolean getBooleanOrThrow(@NotNull String key) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Boolean b -> b;
			case String s -> Boolean.parseBoolean(s);
			case null -> throw new IllegalArgumentException("Boolean value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Boolean value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns a boolean value for the key, or a default value if the key is not found.
	 *
	 * @param key key
	 * @param defaultValue default value
	 * @return boolean or default value
	 */
	public boolean getBooleanOrDefault(@NotNull String key, boolean defaultValue) {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Boolean b -> b;
			case String s -> Boolean.parseBoolean(s);
			case null, default -> defaultValue;
		};
	}

	/**
	 * Returns the byte value by key.
	 *
	 * @param key key
	 * @return byte value
	 * @throws IllegalArgumentException if value is not found or cannot be converted to byte type
	 */
	public byte getByteOrThrow(@NotNull String key) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Number n -> n.byteValue();
			case String s when NumberUtils.isParsable(s) -> Byte.parseByte(s);
			case null -> throw new IllegalArgumentException("Byte value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Byte value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns the byte value for the key, or the default value if the key is not found.
	 *
	 * @param key the key
	 * @param defaultValue the default value
	 * @return the byte value or default value
	 */
	public byte getByteOrDefault(@NotNull String key, byte defaultValue) {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Number n -> n.byteValue();
			case String s when NumberUtils.isParsable(s) -> Byte.parseByte(s);
			case null, default -> defaultValue;
		};
	}

	/**
	 * Returns a short value by key.
	 *
	 * @param key key
	 * @return short value
	 * @throws IllegalArgumentException if value is not found or cannot be converted to short
	 */
	public short getShortOrThrow(@NotNull String key) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Number n -> n.shortValue();
			case String s when NumberUtils.isParsable(s) -> Short.parseShort(s);
			case null -> throw new IllegalArgumentException("Short value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Short value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns a short value by key, or a default value if the key is not found.
	 *
	 * @param key key
	 * @param defaultValue default value
	 * @return short value or default value
	 */
	public short getShortOrDefault(@NotNull String key, short defaultValue) {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Number n -> n.shortValue();
			case String s when NumberUtils.isParsable(s) -> Short.parseShort(s);
			case null, default -> defaultValue;
		};
	}

	/**
	 * Returns an integer value by key.
	 *
	 * @param key key
	 * @return integer value
	 * @throws IllegalArgumentException if value is not found or cannot be converted to integer
	 */
	public int getIntOrThrow(@NotNull String key) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Number n -> n.intValue();
			case String s when NumberUtils.isParsable(s) -> Integer.parseInt(s);
			case null -> throw new IllegalArgumentException("Integer value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Integer value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns an integer value for the key, or a default value if the key is not found.
	 *
	 * @param key key
	 * @param defaultValue default value
	 * @return integer value or default value
	 */
	public int getIntOrDefault(@NotNull String key, int defaultValue) {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Number n -> n.intValue();
			case String s when NumberUtils.isParsable(s) -> Integer.parseInt(s);
			case null, default -> defaultValue;
		};
	}

	/**
	 * Returns a long value by key.
	 *
	 * @param key key
	 * @return long value
	 * @throws IllegalArgumentException if value is not found or cannot be converted to long
	 */
	public long getLongOrThrow(@NotNull String key) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Number n -> n.longValue();
			case String s when NumberUtils.isParsable(s) -> Long.parseLong(s);
			case null -> throw new IllegalArgumentException("Long value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Long value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns a long value by key, or a default value if the key is not found.
	 *
	 * @param key key
	 * @param defaultValue default value
	 * @return long value or default value
	 */
	public long getLongOrDefault(@NotNull String key, long defaultValue) {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Number n -> n.longValue();
			case String s when NumberUtils.isParsable(s) -> Long.parseLong(s);
			case null, default -> defaultValue;
		};
	}

	/**
	 * Returns a float value by key.
	 *
	 * @param key key
	 * @return float value
	 * @throws IllegalArgumentException if value is not found or cannot be converted to float
	 */
	public float getFloatOrThrow(@NotNull String key) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Number n -> n.floatValue();
			case String s when NumberUtils.isParsable(s) -> Float.parseFloat(s);
			case null -> throw new IllegalArgumentException("Float value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Float value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns a float value by key, or a default value if the key is not found.
	 *
	 * @param key key
	 * @param defaultValue default value
	 * @return float value or default value
	 */
	public float getFloatOrDefault(@NotNull String key, float defaultValue) {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Number n -> n.floatValue();
			case String s when NumberUtils.isParsable(s) -> Float.parseFloat(s);
			case null, default -> defaultValue;
		};
	}

	/**
	 * Returns a double value by key.
	 *
	 * @param key key
	 * @return double value
	 * @throws IllegalArgumentException if value is not found or cannot be converted to double
	 */
	public double getDoubleOrThrow(@NotNull String key) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Number n -> n.doubleValue();
			case String s when NumberUtils.isParsable(s) -> Double.parseDouble(s);
			case null -> throw new IllegalArgumentException("Double value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Double value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns a double value by key, or a default value if the key is not found.
	 *
	 * @param key key
	 * @param defaultValue default value
	 * @return double value or default value
	 */
	public double getDoubleOrDefault(@NotNull String key, double defaultValue) {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Number n -> n.doubleValue();
			case String s when NumberUtils.isParsable(s) -> Double.parseDouble(s);
			case null, default -> defaultValue;
		};
	}

	/**
	 * Returns a string value by key.
	 *
	 * @param key key
	 * @return string value
	 * @throws IllegalArgumentException if value not found
	 */
	public @NotNull String getStringOrThrow(@NotNull String key) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case null -> throw new IllegalArgumentException("String value required, but not found: " + key);
			case String s -> s;
			default -> String.valueOf(val);
		};
	}

	/**
	 * Returns a string value for a key, or a default value if the key is not found.
	 *
	 * @param key key
	 * @param defaultValue default value
	 * @return string value or default value
	 */
	public @Nullable String getStringOrDefault(@NotNull String key, @Nullable String defaultValue) {
		final Object val = internalMap.get(key);
		return switch (val) {
			case null -> defaultValue;
			case String s -> s;
			default -> String.valueOf(val);
		};
	}

	/**
	 * Returns an enumeration value by key.
	 *
	 * @param key the key
	 * @param enumClass the enumeration class
	 * @param <T> the enumeration type
	 * @return the enumeration value
	 * @throws IllegalArgumentException if the value is not found, cannot be converted to an enumeration type, or the class is not an enumeration
	 */
	public <T extends Enum<T>> @NotNull T getEnumOrThrow(@NotNull String key, @NotNull Class<T> enumClass) throws IllegalArgumentException {
		if (!enumClass.isEnum()) {
			throw new IllegalArgumentException("Invalid enum class: " + enumClass);
		}
		final Object val = internalMap.get(key);
		return switch (val) {
			case Object o when enumClass.isInstance(o) -> enumClass.cast(o);
			case String s -> Enum.valueOf(enumClass, s);
			case null, default -> throw new IllegalArgumentException("Enum value of type " + enumClass.getName() + " required, but not found: " + key);
		};
	}

	/**
	 * Returns the enumeration value by key, or the default value if the key is not found.
	 *
	 * @param key the key
	 * @param enumClass the enumeration class
	 * @param defaultValue the default value
	 * @param <T> the enumeration type
	 * @return the enumeration value or default value
	 * @throws IllegalArgumentException if the class is not an enum
	 */
	public @Nullable <T extends Enum<T>> T getEnumOrDefault(@NotNull String key, @NotNull Class<T> enumClass, @Nullable T defaultValue, @Nullable String start, @Nullable String end) {
		if (!enumClass.isEnum()) {
			throw new IllegalArgumentException("Invalid enum class: " + enumClass);
		}
		final Object val = internalMap.get(key);
		return switch (val) {
			case Object o when enumClass.isInstance(o) -> enumClass.cast(o);
			case String s -> Enum.valueOf(enumClass, cleanse(s, start, end));
			case null, default -> defaultValue;
		};
	}

	/**
	 * Returns a raw object by key.
	 *
	 * @param key key
	 * @return raw object or null if key not found
	 */
	public @Nullable Object getRawObject(@NotNull String key) {
		return internalMap.get(key);
	}

	/**
	 * Returns an object by key, cast to the given type.
	 *
	 * @param key key
	 * @param type class to cast the object to
	 * @param <A> object type
	 * @return object cast to the given type
	 * @throws IllegalArgumentException if the value is not found or cannot be cast to the given type
	 */
	public <A> @NotNull A getObjectOrThrow(@NotNull String key, @NotNull Class<A> type) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Object o when type.isInstance(o) -> type.cast(o);
			case null -> throw new IllegalArgumentException("Object value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Object value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns the object by key, cast to the given type, or the default value if the key is not found.
	 *
	 * @param key the key
	 * @param type the class to cast the object to
	 * @param defaultValue the default value
	 * @param <A> the type of the object
	 * @return the object cast to the given type, or the default value
	 */
	public <A> @Nullable A getObjectOrDefault(@NotNull String key, @NotNull Class<A> type, @Nullable A defaultValue) {
		final Object val = internalMap.get(key);
		return switch (val) {
			case Object o when type.isInstance(o) -> type.cast(o);
			case null, default -> defaultValue;
		};
	}

	/**
	 * Returns an array of booleans by key.
	 *
	 * @param key key
	 * @param splitOn delimiter for parsing string into array
	 * @param start start character to clear string
	 * @param end end character to clear string
	 * @return array of booleans
	 * @throws IllegalArgumentException if value is not found or cannot be converted to boolean array
	 */
	public boolean @NotNull [] getParsedBooleanArrayOrThrow(@NotNull String key, @NotNull String splitOn, @Nullable String start, @Nullable String end) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case boolean[] v -> v;
			case Boolean[] v -> ArrayUtils.toPrimitive(v);
			case Boolean v -> new boolean[]{v};
			case String s -> ArrayUtils.toPrimitive(Arrays.stream(cleanse(s, start, end).split(splitOn)).map(Boolean::parseBoolean).toArray(Boolean[]::new));
			case null -> throw new IllegalArgumentException("Boolean value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Boolean value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns a char array by key.
	 *
	 * @param key key
	 * @param start start character to clear the string
	 * @param end end character to clear the string
	 * @return char array
	 * @throws IllegalArgumentException if value is not found or cannot be converted to char array
	 */
	public char @NotNull [] getParsedCharArrayOrThrow(@NotNull String key, @Nullable String start, @Nullable String end) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case char[] v -> v;
			case Character[] v -> ArrayUtils.toPrimitive(v);
			case Character v -> new char[]{v};
			case String s -> cleanse(s, start, end).toCharArray();
			case null -> throw new IllegalArgumentException("Character value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Character value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns a byte array by key.
	 *
	 * @param key the key
	 * @param splitOn the separator for parsing the string into an array
	 * @param start the starting character to clear the string
	 * @param end the ending character to clear the string
	 * @return the byte array
	 * @throws IllegalArgumentException if the value is not found or cannot be converted to a byte array
	 */
	public byte @NotNull [] getParsedByteArrayOrThrow(@NotNull String key, @NotNull String splitOn, @Nullable String start, @Nullable String end) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case byte[] v -> v;
			case Byte[] v -> ArrayUtils.toPrimitive(v);
			case Byte v -> new byte[]{v};
			case String s -> ArrayUtils.toPrimitive(Arrays.stream(cleanse(s, start, end).split(splitOn)).map(Byte::parseByte).toArray(Byte[]::new));
			case null -> throw new IllegalArgumentException("Byte value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Byte value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns an array of short values ​​by key.
	 *
	 * @param key key
	 * @param splitOn delimiter for parsing string into array
	 * @param start start character to clear string
	 * @param end end character to clear string
	 * @return array of short values
	 * @throws IllegalArgumentException if value is not found or cannot be converted to array of short values
	 */
	public short @NotNull [] getParsedShortArrayOrThrow(@NotNull String key, @NotNull String splitOn, @Nullable String start, @Nullable String end) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case short[] a -> a;
			case Short[] a -> ArrayUtils.toPrimitive(a);
			case Short v -> new short[]{v};
			case String s -> ArrayUtils.toPrimitive(Arrays.stream(cleanse(s, start, end).split(splitOn)).map(Short::parseShort).toArray(Short[]::new));
			case null -> throw new IllegalArgumentException("Short value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Short value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns an array of integers by key.
	 *
	 * @param key key
	 * @param splitOn separator for parsing string to array
	 * @param start start character to clear string
	 * @param end end character to clear string
	 * @return array of integers
	 * @throws IllegalArgumentException if value is not found or cannot be converted to array of integers
	 */
	public int @NotNull [] getParsedIntArrayOrThrow(@NotNull String key, @NotNull String splitOn, @Nullable String start, @Nullable String end) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case int[] a -> a;
			case Integer[] a -> ArrayUtils.toPrimitive(a);
			case Integer v -> new int[]{v};
			case String s -> Arrays.stream(cleanse(s, start, end).split(splitOn)).mapToInt(Integer::parseInt).toArray();
			case null -> throw new IllegalArgumentException("Integer value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Integer value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns an array of long values ​​by key.
	 *
	 * @param key key
	 * @param splitOn separator for parsing string into array
	 * @param start start character to clear string
	 * @param end end character to clear string
	 * @return array of long values
	 * @throws IllegalArgumentException if value is not found or cannot be converted to array of long values
	 */
	public long @NotNull [] getParsedLongArrayOrThrow(@NotNull String key, @NotNull String splitOn, @Nullable String start, @Nullable String end) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case long[] a -> a;
			case Long[] a -> ArrayUtils.toPrimitive(a);
			case Long v -> new long[]{v};
			case String s -> Arrays.stream(cleanse(s, start, end).split(splitOn)).mapToLong(Long::parseLong).toArray();
			case null -> throw new IllegalArgumentException("Long value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Long value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns an array of floats by key.
	 *
	 * @param key key
	 * @param splitOn separator for parsing string to array
	 * @param start start character to clear string
	 * @param end end character to clear string
	 * @return array of floats
	 * @throws IllegalArgumentException if value is not found or cannot be converted to array of floats
	 */
	public float @NotNull [] getParsedFloatArrayOrThrow(@NotNull String key, @NotNull String splitOn, @Nullable String start, @Nullable String end) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case float[] a -> a;
			case Float[] a -> ArrayUtils.toPrimitive(a);
			case Float v -> new float[]{v};
			case String s -> ArrayUtils.toPrimitive(Arrays.stream(cleanse(s, start, end).split(splitOn)).map(Float::parseFloat).toArray(Float[]::new));
			case null -> throw new IllegalArgumentException("Float value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Float value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Returns an array of doubles by key.
	 *
	 * @param key key
	 * @param splitOn separator for parsing string into array
	 * @param start start character to clear string
	 * @param end end character to clear string
	 * @return array of doubles
	 * @throws IllegalArgumentException if value is not found or cannot be converted to array of doubles
	 */
	public double @NotNull [] getParsedDoubleArrayOrThrow(@NotNull String key, @NotNull String splitOn, @Nullable String start, @Nullable String end) throws IllegalArgumentException {
		final Object val = internalMap.get(key);
		return switch (val) {
			case double[] a -> a;
			case Double[] a -> ArrayUtils.toPrimitive(a);
			case Double v -> new double[]{v};
			case String s -> Arrays.stream(cleanse(s, start, end).split(splitOn)).mapToDouble(Double::parseDouble).toArray();
			case null -> throw new IllegalArgumentException("Double value required, but not found: " + key);
			default -> throw new IllegalArgumentException("Double value required, but not specified: " + key + " = " + val);
		};
	}

	/**
	 * Puts an object into the map by key if the value is not null.
	 *
	 * @param key key
	 * @param value value
	 */
	public void putObject(@NotNull String key, @Nullable Object value) {
		if (value == null) {
			return;
		}
		internalMap.put(key, value);
	}

	/**
	 * Puts a boolean value into the map by key.
	 *
	 * @param key key
	 * @param value value
	 */
	public void putBoolean(@NotNull String key, boolean value) {
		internalMap.put(key, value ? Boolean.TRUE : Boolean.FALSE);
	}

	/**
	 * Puts a byte value into the map by key.
	 *
	 * @param key key
	 * @param value value
	 */
	public void putByte(@NotNull String key, byte value) {
		internalMap.put(key, value);
	}

	/**
	 * Puts a short value into the map by key.
	 *
	 * @param key key
	 * @param value value
	 */
	public void putShort(@NotNull String key, short value) {
		internalMap.put(key, value);
	}

	/**
	 * Puts an integer value into the map by key.
	 *
	 * @param key key
	 * @param value value
	 */
	public void putInt(@NotNull String key, int value) {
		internalMap.put(key, value);
	}

	/**
	 * Puts a long value into the map by key.
	 *
	 * @param key key
	 * @param value value
	 */
	public void putLong(@NotNull String key, long value) {
		internalMap.put(key, value);
	}

	/**
	 * Puts a float value into the map by key.
	 *
	 * @param key key
	 * @param value value
	 */
	public void putFloat(@NotNull String key, float value) {
		internalMap.put(key, value);
	}

	/**
	 * Puts a double value into the map by key.
	 *
	 * @param key key
	 * @param value value
	 */
	public void putDouble(@NotNull String key, double value) {
		internalMap.put(key, value);
	}

	/**
	 * Puts a string value into the map by key if the value is not null.
	 *
	 * @param key key
	 * @param value value
	 */
	public void putString(@NotNull String key, @Nullable String value) {
		if (value == null) {
			return;
		}
		internalMap.put(key, value);
	}

	/**
	 * Puts a string value into the map by key if the value is not null.
	 *
	 * @param key key
	 * @param value value
	 */
	public void putEnum(@NotNull String key, @Nullable Enum<?> value) {
		if (value == null) {
			return;
		}
		internalMap.put(key, value);
	}

	/**
	 * Returns a string representation of the map.
	 *
	 * @return a string representation of the map
	 */
	@Override
	public String toString() {
		return "MultiTypesMap{wrapped=" + internalMap + "}";
	}

	/**
	 * Returns the set of map keys.
	 *
	 * @return the set of map keys
	 */
	@Override
	public @NotNull Set<String> keySet() {
		return internalMap.keySet();
	}

	/**
	 * Returns a collection of map values.
	 *
	 * @return a collection of map values
	 */
	@Override
	public @NotNull Collection<Object> values() {
		return internalMap.values();
	}

	/**
	 * Returns the map recordset.
	 *
	 * @return the map recordset
	 */
	@Override
	public @NotNull Set<Entry<String, Object>> entrySet() {
		return internalMap.entrySet();
	}

	/**
	 * Clears the map.
	 */
	@Override
	public void clear() {
		internalMap.clear();
	}

	/**
	 * Returns the size of the map.
	 *
	 * @return the size of the map
	 */
	@Override
	public int size() {
		return internalMap.size();
	}

	/**
	 * Checks if the map is empty.
	 *
	 * @return true if the map is empty, false otherwise
	 */
	@Override
	public boolean isEmpty() {
		return internalMap.isEmpty();
	}

	/**
	 * Checks if the map contains the key.
	 *
	 * @param key key
	 * @return true if the map contains the key, false otherwise
	 */
	@Override
	public boolean containsKey(Object key) {
		return internalMap.containsKey(key);
	}

	/**
	 * Checks if the map contains a value.
	 *
	 * @param value value
	 * @return true if the map contains a value, false otherwise
	 */
	@Override
	public boolean containsValue(Object value) {
		return internalMap.containsValue(value);
	}

	/**
	 * Returns the value by key.
	 *
	 * @param key key
	 * @return the value by key or null if the key is not found
	 */
	@Override
	public @Nullable Object get(Object key) {
		return internalMap.get(key);
	}

	/**
	 * Puts a value into the map by key.
	 *
	 * @param key key
	 * @param value value
	 * @return previous value by key or null if key was not found
	 */
	@Override
	public @Nullable Object put(String key, Object value) {
		return internalMap.put(key, value);
	}

	/**
	 * Copies all records from another map to the current one.
	 *
	 * @param map the map from which to copy records
	 */
	@Override
	public void putAll(@NotNull Map<? extends String, ?> map) {
		if (Objects.equals(internalMap, map) || map.isEmpty()) {
			return;
		}
		internalMap.putAll(map);
	}

	/**
	 * Deletes a record by key.
	 *
	 * @param key key
	 * @return the deleted value or null if the key was not found
	 */
	@Override
	public Object remove(Object key) {
		return internalMap.remove(key);
	}

	/**
	 * Returns the value by key, or the default value if the key is not found.
	 *
	 * @param key key
	 * @param defaultValue default value
	 * @return the value by key or default value
	 */
	@Override
	public Object getOrDefault(Object key, Object defaultValue) {
		return internalMap.getOrDefault(key, defaultValue);
	}

	/**
	 * Performs the given action for each entry in the map.
	 *
	 * @param action the action to perform
	 */
	@Override
	public void forEach(BiConsumer<? super String, ? super Object> action) {
		internalMap.forEach(action);
	}

	/**
	 * Replaces all values ​​in the map with the results of calling the given function.
	 *
	 * @param function the function to apply to the values
	 */
	@Override
	public void replaceAll(BiFunction<? super String, ? super Object, ?> function) {
		internalMap.replaceAll(function);
	}

	/**
	 * Puts a value into the map by key if the key is missing.
	 *
	 * @param key key
	 * @param value value
	 * @return previous value by key or null if the key was not found
	 */
	@Override
	public @Nullable Object putIfAbsent(String key, Object value) {
		return internalMap.putIfAbsent(key, value);
	}

	/**
	 * Deletes a record by key and value.
	 *
	 * @param key key
	 * @param value value
	 * @return true if the record was deleted, otherwise false
	 */
	@Override
	public boolean remove(Object key, Object value) {
		return internalMap.remove(key, value);
	}

	/**
	 * Replaces a value by key
	 *
	 * @param key key
	 * @param oldValue old value
	 * @param newValue new value
	 * @return true if the value was replaced, otherwise false
	 */
	@Override
	public boolean replace(String key, Object oldValue, Object newValue) {
		return internalMap.replace(key, oldValue, newValue);
	}

	/**
	 * Replaces the value by key.
	 *
	 * @param key key
	 * @param value value
	 * @return previous value by key or null if key was not found
	 */
	@Override
	public @Nullable Object replace(String key, Object value) {
		return internalMap.replace(key, value);
	}

	/**
	 * Calculates the value for the key if the key is missing, using the given function.
	 *
	 * @param key the key
	 * @param mappingFunction the function to calculate the value
	 * @return the calculated value, or null if the function returned null
	 */
	@Override
	public Object computeIfAbsent(String key, @NotNull Function<? super String, ?> mappingFunction) {
		return internalMap.computeIfAbsent(key, mappingFunction);
	}

	/**
	 * Computes a new value for the key, if the key is present, using the given function.
	 *
	 * @param key the key
	 * @param remappingFunction the function to compute the new value
	 * @return the new value, or null if the function returned null
	 */
	@Override
	public Object computeIfPresent(String key, @NotNull BiFunction<? super String, ? super Object, ?> remappingFunction) {
		return internalMap.computeIfPresent(key, remappingFunction);
	}

	/**
	 * Calculates a new value for the key using the given function.
	 *
	 * @param key the key
	 * @param remappingFunction the function to calculate the new value
	 * @return the new value or null if the function returned null
	 */
	@Override
	public Object compute(String key, @NotNull BiFunction<? super String, ? super @Nullable Object, ?> remappingFunction) {
		return internalMap.compute(key, remappingFunction);
	}

	/**
	 * Merge a value by key with a given value using a given function.
	 *
	 * @param key the key
	 * @param value the value
	 * @param remappingFunction the function to merge the values
	 * @return the new value or null if the function returned null
	 */
	@Override
	public Object merge(String key, @NotNull Object value, @NotNull BiFunction<? super Object, ? super Object, ?> remappingFunction) {
		return internalMap.merge(key, value, remappingFunction);
	}

	/**
	 * Removes leading and trailing substrings from a string, if present.
	 *
	 * @param string string
	 * @param start starting substring
	 * @param end ending substring
	 * @return stripped string
	 */
	private static String cleanse(String string, String start, String end) {
		if (string != null && start != null && end != null) {
			if (string.startsWith(start) && string.endsWith(end)) {
				string = string.substring(start.length(), string.length() - end.length());
			}
		}
		return string;
	}
}