package host.anzo.commons.concurrent;

import lombok.extern.log4j.Log4j2;
import org.jetbrains.annotations.NotNull;

import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * Extension of {@link ReentrantReadWriteLock} providing convenience methods for try-with-resources lock management.
 * <p>
 * Wraps readLock and writeLock with {@link AutoCloseable} implementations, enabling automatic lock release
 * when exiting try-with-resources blocks.
 * <p>
 * Usage examples with {@code var} and unnamed variables in try-with-resources:
 *
 * <pre>{@code
 * // Create lock instance
 * var lock = new CloseableReentrantReadWriteLock(false);
 *
 * // Read lock with unnamed variable
 * try (var _ = lock.readOpen()) {
 *     // Read protected data
 *     System.out.println("Reading data...");
 * } // Read lock auto-released here
 *
 * // Write lock with named variable
 * try (var writeLock = lock.writeOpen()) {
 *     // Modify protected data
 *     System.out.println("Writing data...");
 * } // Write lock auto-released here
 *
 * // Multiple locks with unnamed variables
 * try (var _ = lock.readLock();
 *      var __ = lock.writeLock()) {
 *     // Combined read/write operations
 *     System.out.println("Reading and writing data...");
 * }
 * }</pre>
 *
 * @author Aristo
 * @since 20.12.2023
 */
@Log4j2
public class CloseableReentrantReadWriteLock extends ReentrantReadWriteLock {
	private final CloseableReentrantReadLock readLock;
	private final CloseableReentrantWriteLock writeLock;

	/**
	 * Creates a new {@link CloseableReentrantReadWriteLock} with specified fairness policy.
	 *
	 * @param fair {@code true} for fair ordering policy, {@code false} for non-fair
	 */
	public CloseableReentrantReadWriteLock(boolean fair) {
		super(fair);
		readLock = new CloseableReentrantReadLock(this);
		writeLock = new CloseableReentrantWriteLock(this);
	}

	/**
	 * AutoCloseable wrapper for {@link ReadLock} implementing try-with-resources pattern.
	 */
	public static final class CloseableReentrantReadLock extends ReadLock implements AutoCloseable {

		public CloseableReentrantReadLock(ReentrantReadWriteLock lock) {
			super(lock);
		}

		/**
		 * Acquires the read lock and returns self-reference for resource management.
		 *
		 * @return this instance for chaining
		 */
		public CloseableReentrantReadLock readOpen() {
			this.lock();
			return this;
		}

		/**
		 * Releases the read lock. Handles double-unlock attempts gracefully.
		 */
		@Override
		public void close() {
			try {
				this.unlock();
			} catch (IllegalMonitorStateException e) {
				log.warn("Attempted to unlock an unlocked ReadLock", e);
			}
		}
	}

	/**
	 * AutoCloseable wrapper for {@link WriteLock} implementing try-with-resources pattern.
	 */
	public static final class CloseableReentrantWriteLock extends WriteLock implements AutoCloseable {

		public CloseableReentrantWriteLock(ReentrantReadWriteLock lock) {
			super(lock);
		}

		/**
		 * Acquires the write lock and returns self-reference for resource management.
		 *
		 * @return this instance for chaining
		 */
		public CloseableReentrantWriteLock writeOpen() {
			this.lock();
			return this;
		}

		/**
		 * Releases the write lock. Handles double-unlock attempts gracefully.
		 */
		@Override
		public void close() {
			try {
				this.unlock();
			} catch (IllegalMonitorStateException e) {
				log.warn("Attempted to unlock an unlocked WriteLock", e);
			}
		}
	}

	/**
	 * Acquires read lock and returns AutoCloseable wrapper.
	 *
	 * @return {@link CloseableReentrantReadLock} that auto-releases when closed
	 */
	public CloseableReentrantReadLock readOpen() {
		return this.readLock.readOpen();
	}

	/**
	 * Acquires write lock and returns AutoCloseable wrapper.
	 *
	 * @return {@link CloseableReentrantWriteLock} that auto-releases when closed
	 */
	public CloseableReentrantWriteLock writeOpen() {
		return this.writeLock.writeOpen();
	}

	/**
	 * Returns the closeable read lock wrapper.
	 *
	 * @return {@link CloseableReentrantReadLock} instance
	 */
	@Override
	public @NotNull CloseableReentrantReadLock readLock() {
		return readLock;
	}

	/**
	 * Returns the closeable write lock wrapper.
	 *
	 * @return {@link CloseableReentrantWriteLock} instance
	 */
	@Override
	public @NotNull CloseableReentrantWriteLock writeLock() {
		return writeLock;
	}
}