package host.anzo.commons.utils;

import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.Unmodifiable;

import java.io.File;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.*;

/**
 * Utility class for performing various operations related to Java classes, including
 * singleton instance retrieval, method and field reflection, and dynamic class loading.
 * @author ANZO
 * @since 12.09.2013
 */
@Slf4j
public class ClassUtils {
	/**
	 * Retrieves the singleton instance of the specified class by invoking its static
	 * "getInstance" method.
	 * @param clazz the class to retrieve the singleton instance from
	 * @return the singleton instance of the specified class
	 */
	public static Object singletonInstance(Class<?> clazz) {
		try {
			Method method = clazz.getDeclaredMethod("getInstance");
			return method.invoke(null);
		} catch (Exception e) {
			e.printStackTrace();
			System.exit(0);
			throw null;
		}
	}

	/**
	 * Invokes a specified method on the singleton instance of the given class.
	 *
	 * @param clazz the class containing the singleton instance
	 * @param method the method to invoke on the singleton instance
	 * @return the result of the method invocation
	 */
	public static Object singletonInstanceMethod(Class<?> clazz, Method method) {
		try {
			final Object singletonInstance = singletonInstance(clazz);
			if (singletonInstance != null) {
				return method.invoke(singletonInstance);
			}
			return null;
		} catch (Exception e) {
			e.printStackTrace();
			System.exit(0);
			return null;
		}
	}

	/**
	 * Retrieves all methods annotated with a specified annotation from the given class
	 * and its superclasses and interfaces.
	 *
	 * @param type the class to inspect for annotated methods
	 * @param annotation the annotation to look for
	 * @return a collection of methods annotated with the specified annotation
	 */
	public static @NotNull Collection<Method> getMethodsAnnotatedWith(final Class<?> type, final Class<? extends Annotation> annotation) {
		final Map<String, Method> methods = new HashMap<>();
		Class<?> currentClass = type;
		while (currentClass != Object.class) {
			collectMethodsAnnotatedWith(currentClass, type, annotation, methods);
			currentClass = currentClass.getSuperclass();
		}
		final Class<?>[] classInterfaces = type.getInterfaces();
		for (Class<?> classInterface : classInterfaces) {
			collectMethodsAnnotatedWith(classInterface, type, annotation, methods);
		}
		return methods.values();
	}

	/**
	 * Collects methods annotated with a specified annotation from a given class.
	 *
	 * @param klass the class to inspect for annotated methods
	 * @param baseKlass the base class for logging purposes
	 * @param annotation the annotation to look for
	 * @param methods a map to store found methods
	 */
	private static void collectMethodsAnnotatedWith(final @NotNull Class<?> klass, final Class<?> baseKlass, final Class<? extends Annotation> annotation, Map<String, Method> methods) {
		final List<Method> allMethods = Arrays.stream(klass.getDeclaredMethods()).filter(method -> method.isAnnotationPresent(annotation)).toList();
		for (Method method : allMethods) {
			if (!methods.containsKey(method.getName())) {
				methods.put(method.getName(), method);
			} else {
				log.error("Class [{}] contains duplicate method [{}] (original class [{}])", klass.getName(), method.getName(), baseKlass);
			}
		}
	}

	/**
	 * Attempts to load a class by its name quietly, returning null if the class cannot be found.
	 *
	 * @param name the name of the class to load
	 * @return the loaded class, or null if not found
	 */
	public static @Nullable Class<?> lookupClassQuietly(String name) {
		try {
			return ClassUtils.class.getClassLoader().loadClass(name);
		} catch (Exception ignored) {
		}
		return null;
	}

	/**
	 * Retrieves the values of all static fields in the specified class.
	 *
	 * @param clazz the class to inspect for field values
	 * @return a map of field names to their corresponding values
	 */
	public static @NotNull Map<String, Object> getFieldValues(final @NotNull Class<?> clazz) {
		final Map<String, Object> fieldValues = new HashMap<>();
		for (Field field : clazz.getDeclaredFields()) {
			try {
				fieldValues.put(field.getName(), field.get(null));
			} catch (IllegalAccessException e) {
				e.printStackTrace();
			}
		}
		return fieldValues;
	}

	/**
	 * Initializes the fields of a given object using values from a provided map.
	 *
	 * @param object the object to initialize
	 * @param fieldValues a map of field names and their corresponding values
	 */
	public static void initFieldsFromMap(final Object object, @NotNull Map<String, Object> fieldValues) {
		for (Map.Entry<String, Object> entry : fieldValues.entrySet()) {
			final String fieldName = entry.getKey();
			final Object fieldValue = entry.getValue();
			final Field objectField;
			try {
				objectField = object.getClass().getField(fieldName);
				objectField.set(object, fieldValue);
			} catch (IllegalAccessException e) {
				e.printStackTrace();
			} catch (NoSuchFieldException ignored) {
			}
		}
	}

	/**
	 * Retrieves a constructor of the specified class that matches the given parameter types.
	 *
	 * @param clazz the class to inspect for constructors
	 * @param parameterTypes the parameter types to match
	 * @param <T> the type of the class
	 * @return the matching constructor, or null if not found
	 */
	public static @Nullable <T> Constructor<T> getConstructor(final @NotNull Class<T> clazz, Class<?>... parameterTypes) {
		for (Constructor<?> constructor : clazz.getConstructors()) {
			if (constructor.getParameterTypes().length != parameterTypes.length) {
				continue;
			}
			boolean paramsMatch = true;
			for (int paramIndex = 0; paramIndex < parameterTypes.length; paramIndex++) {
				if (!constructor.getParameterTypes()[paramIndex].equals(parameterTypes[paramIndex])) {
					paramsMatch = false;
					break;
				}
			}
			if (paramsMatch) {
				return (Constructor<T>) constructor;
			}
		}
		return null;
	}

	/**
	 * Checks if the specified class has a constructor that matches the given parameter types.
	 *
	 * @param clazz the class to inspect for constructors
	 * @param parameterTypes the parameter types to match
	 * @return true if a matching constructor exists, false otherwise
	 */
	public static boolean hasConstructor(final @NotNull Class<?> clazz, Class<?>... parameterTypes) {
		return getConstructor(clazz, parameterTypes) != null;
	}

	/**
	 * Adds JAR files to the class loader dynamically.
	 *
	 * @param jarFiles a list of JAR files to add
	 * @return the updated ClassLoader
	 * @throws IOException if an I/O error occurs while adding JARs
	 */
	public static @NotNull ClassLoader addJars(@NotNull List<File> jarFiles) throws IOException {
		final URL[] jarUrls = new URL[jarFiles.size()];
		for (int i = 0; i < jarFiles.size(); i++) {
			jarUrls[i] = jarFiles.get(i).toURI().toURL();
		}

		final ClassLoader systemClassLoader = ClassLoader.getSystemClassLoader();
		if (!(systemClassLoader instanceof URLClassLoader)) {
			final URLClassLoader urlClassLoader = new URLClassLoader(jarUrls, systemClassLoader);
			Thread.currentThread().setContextClassLoader(urlClassLoader);
			return urlClassLoader;
		}

		return systemClassLoader;
	}

	/**
	 * Retrieves all the constants of the specified enum class as an unmodifiable list.
	 *
	 * @param enumClass the class object of the enum type from which to retrieve the constants
	 * @return a list containing all the constants of the specified enum class
	 * @throws IllegalArgumentException if the provided class is not an enum
	 */
	@SuppressWarnings("unchecked")
	public static @NotNull @Unmodifiable List<Enum<?>> getEnumValues(@NotNull Class<?> enumClass) {
		if (!enumClass.isEnum()) {
			throw new IllegalArgumentException("Provided class is not an enum: " + enumClass.getName());
		}
		return List.of(((Class<Enum<?>>) enumClass).getEnumConstants());
	}
}