package host.anzo.commons.emergency.memory;

import com.sun.management.GarbageCollectionNotificationInfo;
import com.sun.management.GcInfo;
import de.mxro.metrics.jre.Metrics;
import delight.async.properties.PropertyNode;
import host.anzo.commons.annotations.startup.Scheduled;
import host.anzo.commons.annotations.startup.StartupComponent;
import host.anzo.commons.emergency.memory.watchers.DefaultWatcher;
import host.anzo.commons.emergency.memory.watchers.G1Watcher;
import host.anzo.commons.emergency.memory.watchers.IMemoryWatcher;
import host.anzo.commons.emergency.metric.IMetric;
import host.anzo.commons.emergency.metric.Metric;
import host.anzo.commons.emergency.metric.MetricGroupType;
import host.anzo.commons.emergency.metric.MetricResult;
import host.anzo.commons.utils.VMUtils;
import host.anzo.core.config.EmergencyConfig;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.text.TextStringBuilder;
import org.jetbrains.annotations.NotNull;

import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.openmbean.CompositeData;
import java.lang.management.*;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * @author ANZO
 */
@Slf4j
@Metric
@StartupComponent("Diagnostic")
public class MemoryLeakDetector implements NotificationListener, IMetric {
	@Getter(lazy=true)
	private final static MemoryLeakDetector instance = new MemoryLeakDetector();

	private final static PropertyNode metrics = Metrics.create();
	private final List<IMemoryListener> listeners = new ArrayList<>();

	private MemoryLeakDetector() {
		final List<IMemoryWatcher> watchers = Arrays.asList(new G1Watcher(), new DefaultWatcher());
		for(MemoryPoolMXBean pool : ManagementFactory.getMemoryPoolMXBeans()) {
			watchers.stream()
					.filter(watcher -> watcher.isValid(pool))
					.forEach(watcher -> watcher.register(pool));
			log.info("Watcher for memory pool [{}] successfully registered.", pool.getName());
		}
		for (GarbageCollectorMXBean gcBean : ManagementFactory.getGarbageCollectorMXBeans()) {
			try {
				ManagementFactory.getPlatformMBeanServer().
						addNotificationListener(gcBean.getObjectName(), this, null,null);
			} catch (Exception e) {
				log.error("Error while register GC notification listener", e);
			}
		}
	}

	@SuppressWarnings("unused")
	@Scheduled(period = 10, timeUnit = TimeUnit.MINUTES, runAfterServerStart = true)
	public void reportMemoryStats() {
		for(String memStat : getMemoryUsageStatistics()) {
			log.info(memStat);
		}
		log.info("Memory Pool Information:");
		for(MemoryPoolMXBean pool : ManagementFactory.getMemoryPoolMXBeans()) {
			log.info("- {} : {}", pool.getName(), pool.getUsage().toString());
		}
		log.info("Bytebuffer Pool Information:");
		for (BufferPoolMXBean pool : VMUtils.getByteBufferPools()) {
			log.info("- {}: count={} usedMemory={} totalCapacity={}", pool.getName(), pool.getCount(), pool.getMemoryUsed(), pool.getTotalCapacity());
		}
	}

	@Override
	public void handleNotification(@NotNull Notification notification, Object handback) {
		if (notification.getType().equals(GarbageCollectionNotificationInfo.GARBAGE_COLLECTION_NOTIFICATION)) {
			if (EmergencyConfig.ENABLE_METRICS) {
				final CompositeData cd = (CompositeData) notification.getUserData();
				final GarbageCollectionNotificationInfo gcNotificationInfo = GarbageCollectionNotificationInfo.from(cd);
				final GcInfo gcInfo = gcNotificationInfo.getGcInfo();
				final String gcActionName = gcNotificationInfo.getGcAction().replace(" ", "_");
				metrics.record(Metrics.happened("gc_" + gcActionName + "_count"));
				metrics.record(Metrics.value("gc_" + gcActionName + "_time", gcInfo.getDuration()));
				metrics.record(Metrics.value("gc_" + gcActionName + "_cleaned_mb", Math.max(0, sumUsedMb(gcInfo.getMemoryUsageBeforeGc()) - sumUsedMb(gcInfo.getMemoryUsageAfterGc()))));
			}
		}
	}

	private static long sumUsedMb(@NotNull Map<String, MemoryUsage> memUsages) {
		long sum = 0;
		for (MemoryUsage memoryUsage : memUsages.values()) {
			sum += memoryUsage.getUsed();
		}
		return sum / (1024 * 1024);
	}

	public void registerListener(IMemoryListener listener) {
		listeners.add(listener);
	}

	public void onMemoryLeakDetected() {
		listeners.forEach(IMemoryListener::onMemoryLeakDetected);
	}

	public String getStats() {
		final TextStringBuilder builder = new TextStringBuilder();
		for(MemoryPoolMXBean pool : ManagementFactory.getMemoryPoolMXBeans()) {
			builder.appendln(pool.getName() + " : " + pool.getUsage().toString());
		}
		return builder.get();
	}

	private final SimpleDateFormat timeFormat = new SimpleDateFormat("H:mm:ss");
	private final DecimalFormat percentFormat = new DecimalFormat(" (0.0000'%')");
	private final DecimalFormat sizeFormat = new DecimalFormat(" # 'KB'");

	public String @NotNull [] getMemoryUsageStatistics() {
		final double max = Runtime.getRuntime().maxMemory() / 1024.0; // maxMemory is the upper limit the jvm can use
		final double allocated = Runtime.getRuntime().totalMemory() / 1024.0; //totalMemory the size of the current allocation pool
		final double nonAllocated = max - allocated; //non allocated memory till jvm limit
		final double cached = Runtime.getRuntime().freeMemory() / 1024.0; // freeMemory the unused memory in the allocation pool
		final double used = allocated - cached; // really used memory
		final double usable = max - used; // allocated, but non-used and non-allocated memory

		return new String[] {
				"+----",
				"| Global Memory Information at " + timeFormat.format(new Date()) + ":",
				"|    |",
				"| Allowed Memory:" + sizeFormat.format(max),
				"|    |= Allocated Memory:" + sizeFormat.format(allocated) + percentFormat.format(allocated / max * 100),
				"|    |= Non-Allocated Memory:" + sizeFormat.format(nonAllocated) + percentFormat.format(nonAllocated / max * 100),
				"| Allocated Memory:" + sizeFormat.format(allocated),
				"|    |= Used Memory:" + sizeFormat.format(used) + percentFormat.format(used / max * 100),
				"|    |= Unused (cached) Memory:" + sizeFormat.format(cached) + percentFormat.format(cached / max * 100),
				"| Usable Memory:" + sizeFormat.format(usable) + percentFormat.format(usable / max * 100),
				"+----"};
	}

	@Override
	public List<MetricResult> getMetric() {
		final MetricResult result = new MetricResult();
		result.setMetricGroupType(MetricGroupType.SYSTEM);
		result.setName("MemoryService");
		result.setData(metrics.render().get());
		return Collections.singletonList(result);
	}
}