package host.anzo.commons.collection;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * A utility class that splits a collection into smaller chunks of a specified size.
 * @author xTz, ANZO
 * @param <T> the type of elements in the collection
 */
public class ListSplitter<T> {
	private List<T> objects;
	private int splitCount;
	private int currentIndex = 0;

	/**
	 * Constructs a ListSplitter with the specified collection and split count.
	 *
	 * @param collection the collection to be split
	 * @param splitCount the maximum size of each chunk
	 */
	public ListSplitter(Collection<T> collection, int splitCount) {
		if (collection != null) {
			this.splitCount = splitCount;
			if (collection instanceof List<T> list) {
				this.objects = list;
			}
			else {
				this.objects = new ArrayList<>(collection);
			}
		}
	}

	/**
	 * Retrieves the next chunk of elements with the specified split count.
	 *
	 * @param splitCount the maximum size of the next chunk
	 * @return a list containing the next chunk of elements
	 */
	public List<T> getNext(int splitCount) {
		this.splitCount = splitCount;
		return getNext();
	}

	/**
	 * Retrieves the next chunk of elements based on the current split count.
	 *
	 * @return a list containing the next chunk of elements
	 */
	public List<T> getNext() {
		if (splitCount >= objects.size()) {
			currentIndex = objects.size();
			return objects;
		}

		final int chunkSize = Math.min(splitCount, objects.size() - currentIndex);
		final List<T> chunk = objects.subList(currentIndex, currentIndex + chunkSize);
		currentIndex += chunkSize;
		return chunk;
	}

	/**
	 * Returns the total number of elements in the original collection.
	 *
	 * @return the size of the collection
	 */
	public int size() {
		return objects.size();
	}

	/**
	 * Checks if the current chunk is the first chunk.
	 *
	 * @return true if the current chunk is the first chunk, false otherwise
	 */
	public boolean isFirst() {
		return currentIndex <= splitCount;
	}

	/**
	 * Checks if the current chunk is the last chunk.
	 *
	 * @return true if the current chunk is the last chunk, false otherwise
	 */
	public boolean isLast() {
		return currentIndex == objects.size();
	}

	/**
	 * Checks if there are more chunks available to retrieve.
	 *
	 * @return true if there are more chunks, false otherwise
	 */
	public boolean hasNext() {
		return currentIndex < objects.size();
	}
}