package host.anzo.commons.collection;

import java.util.*;

/**
 * A utility class that splits a map into smaller chunks.
 *
 * @param <K> the type of keys maintained by this map
 * @param <V> the type of mapped values
 *
 * @author ANZO
 * @since 6/22/2022
 */
public class MapSplitter<K, V> {
    private TreeMap<K, V> map;
    private List<K> keys;
    private int splitCount;
    private int currentIndex = 0;

    /**
     * Constructs a MapSplitter with the specified map and split count.
     *
     * @param map the map to be split
     * @param splitCount the number of entries in each chunk
     */
    public MapSplitter(Map<K, V> map, int splitCount) {
        if (map != null) {
            this.map = new TreeMap<>(map);
            this.keys = new ArrayList<>(this.map.keySet());
            this.splitCount = splitCount;
        }
    }

    /**
     * Constructs a MapSplitter with the specified map, comparator, and split count.
     *
     * @param map the map to be split
     * @param comparator the comparator to order the keys
     * @param splitCount the number of entries in each chunk
     */
    public MapSplitter(Map<K, V> map, Comparator<? super K> comparator, int splitCount) {
        if (map != null) {
            this.map = new TreeMap<>(comparator);
            this.map.putAll(map);
            this.keys = new ArrayList<>(this.map.keySet());
            this.splitCount = splitCount;
        }
    }

    /**
     * Retrieves the next chunk of the map with the specified split count.
     *
     * @param splitCount the number of entries in the next chunk
     * @return a map containing the next chunk of entries
     */
    public Map<K, V> getNext(int splitCount) {
        this.splitCount = splitCount;
        return getNext();
    }

    /**
     * Retrieves the next chunk of the map.
     *
     * @return a map containing the next chunk of entries
     */
    public Map<K, V> getNext() {
        final Map<K, V> chunk;
        if (currentIndex + splitCount < map.size()) {
            chunk = map.subMap(keys.get(currentIndex), keys.get(currentIndex + splitCount));
        }
        else {
            chunk = map.tailMap(keys.get(currentIndex));
        }
        currentIndex += splitCount;
        return chunk;
    }

    /**
     * Returns the total number of entries in the map.
     *
     * @return the size of the map
     */
    public int size() {
        return map.size();
    }

    /**
     * Checks if the current chunk is the first chunk.
     *
     * @return true if the current chunk is the first chunk, false otherwise
     */
    public boolean isFirst() {
        return currentIndex <= splitCount;
    }

    /**
     * Checks if the current chunk is the last chunk.
     *
     * @return true if the current chunk is the last chunk, false otherwise
     */
    public boolean isLast() {
        return currentIndex == map.size();
    }

    /**
     * Checks if there are more chunks available to retrieve.
     *
     * @return true if there are more chunks, false otherwise
     */
    public boolean hasNext() {
        return currentIndex < map.size();
    }
}