package host.anzo.commons.utils;

import lombok.extern.slf4j.Slf4j;

import java.io.IOException;

/**
 * Utility class for interacting with systemd notifications.
 * This class provides methods to notify systemd about the service status.
 * @author ANZO
 * @since 6/30/2023
 */
@Slf4j
public class SystemdUtils {
	/**
	 * Notifies systemd that the service is ready.
	 * This method sends a notification with the parameter "--ready".
	 */
	public static void notifyReady() {
		notify("--ready");
	}

	/**
	 * Sets the status of the service in systemd.
	 *
	 * @param status the status message to be set for the service.
	 */
	public static void setStatus(String status) {
		notify("--status=" + status);
	}

	/**
	 * Sends a notification to systemd with the specified parameter.<br>
	 * @param param the parameter to be sent to systemd.
	 */
	private static void notify(String param) {
		if (System.getenv("NOTIFY_SOCKET") == null) {
			return;
		}
		try {
			final Process process = new ProcessBuilder("systemd-notify", param)
					.redirectOutput(ProcessBuilder.Redirect.INHERIT)
					.redirectError(ProcessBuilder.Redirect.INHERIT)
					.start();
			final int exitCode = process.waitFor();
			if (exitCode != 0) {
				log.error("Can't notify systemd exitCode={}", exitCode);
			}
		}
		catch (IOException | InterruptedException e) {
			log.error("Can't notify systemd", e);
		}
	}
}