/*
 * Copyright © 2016 BDO-Emu authors. All rights reserved.
 * Viewing, editing, running and distribution of this software strongly prohibited.
 * Author: xTz, Anton Lasevich, Tibald
 */

package host.anzo.core.service;

import host.anzo.commons.annotations.startup.StartupComponent;
import host.anzo.commons.utils.VMUtils;
import host.anzo.core.config.MailConfig;
import jakarta.mail.*;
import jakarta.mail.internet.InternetAddress;
import jakarta.mail.internet.MimeBodyPart;
import jakarta.mail.internet.MimeMessage;
import jakarta.mail.internet.MimeMultipart;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.Collections;
import java.util.Properties;

/**
 * @author ANZO
 */
@Slf4j
@StartupComponent("Service")
public class MailService {
	@Getter(lazy = true)
	private final static MailService instance = new MailService();

	private Session session;

	MailService() {
		if (!MailConfig.ENABLE) {
			return;
		}

		final Properties mailProps = System.getProperties();
		mailProps.put("mail.transport.protocol", "smtp");
		mailProps.put("mail.smtp.host", MailConfig.SMTP_HOST);
		mailProps.put("mail.smtp.port", MailConfig.SMTP_PORT);
		mailProps.put("mail.smtp.socketFactory.port", MailConfig.SMTP_PORT);
		mailProps.put("mail.smtp.socketFactory.class", "javax.net.ssl.SSLSocketFactory");
		mailProps.put("mail.smtp.socketFactory.fallback", "false");
		mailProps.put("mail.smtp.auth", "true");
		session = Session.getInstance(mailProps, new Authenticator() {
			protected PasswordAuthentication getPasswordAuthentication() {
				return (new PasswordAuthentication(MailConfig.USER, MailConfig.PASSWORD));
			}
		});
		if (VMUtils.DEBUG) {
			session.setDebug(true);
		}
	}

	/**
	 * Send mail to the specified email address
	 * @param email email's address
	 * @param subject mail subject
	 * @param messageText mail text
	 */
	public void sendMail(@NotNull String email, @NotNull String subject, @NotNull String messageText) {
		sendMail(Collections.singletonList(email), subject, messageText);
	}

	/**
	 * Send mail to the specified email address list
	 * @param emails email's address collection
	 * @param subject mail subject
	 * @param messageText mail text
	 */
	public void sendMail(@NotNull Collection<String> emails, @NotNull String subject, @NotNull String messageText) {
		if (emails.isEmpty()) {
			return;
		}
		try {
			final MimeMessage message = new MimeMessage(session);
			message.setFrom(new InternetAddress(MailConfig.USER));
			message.setRecipients(Message.RecipientType.TO, InternetAddress.parse(emails.iterator().next()));
			if (emails.size() > 1) {
				message.setRecipients(Message.RecipientType.CC, InternetAddress.parse(String.join(",", emails)));
			}
			message.setSubject(subject, "UTF-8");

			final Multipart messageMultipart = new MimeMultipart();
			final MimeBodyPart htmlPart = new MimeBodyPart();
			htmlPart.setContent(messageText, "text/html");
			messageMultipart.addBodyPart(htmlPart);
			message.setContent(messageMultipart);
			Transport.send(message);
		} catch (Exception e) {
			log.error("Error while sending mail", e);
		}
	}
}