package host.anzo.commons.utils;

import org.jetbrains.annotations.NotNull;

import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * @author ANZO
 */
public class StreamUtils {
    /**
     * Generates a list of integers representing a closed range from the specified start to end values.
     * @param startInclusive The starting value of the range, inclusive.
     * @param endInclusive   The ending value of the range, inclusive.
     * @return A list containing all integers from startInclusive to endInclusive, inclusive.
     */
    public static List<Integer> closedRange(int startInclusive, int endInclusive) {
        return IntStream.rangeClosed(startInclusive, endInclusive)
                .boxed()
                .collect(Collectors.toList());
    }

    /**
     * Sorts a map by its values in either ascending or descending order.
     * @param map           The map to be sorted, where keys are of type K and values are of type V.
     * @param isReverseOrder A boolean indicating whether to sort in reverse (descending) order.
     * @param <K>          The type of keys in the map.
     * @param <V>          The type of values in the map, which must be comparable.
     * @return A new LinkedHashMap containing the entries of the original map sorted by value.
     */
    public static <K, V extends Comparable<V>> Map<K, V> sortMapByValue(@NotNull Map<K, V> map, boolean isReverseOrder) {
        return map.entrySet().stream()
                .sorted(isReverseOrder ? Map.Entry.comparingByValue(Comparator.reverseOrder()) : Map.Entry.comparingByValue(Comparator.naturalOrder()))
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, (e1, e2) -> e1, LinkedHashMap::new));
    }
}