package host.anzo.commons.concurrent;

import java.util.concurrent.locks.ReentrantLock;

/**
 * A CloseableReentrantLock is a specialized implementation of a ReentrantLock
 * that implements the AutoCloseable interface. This allows the lock to be used
 * in a try-with-resources statement, ensuring that the lock is properly released
 * when it is no longer needed.
 * <br><br>
 * The default fairness policy of the lock is false, meaning that threads will
 * acquire the lock in the order of their arrival.
 * <br><br>
 * If the fairness policy is set to true, threads will acquire the lock in the
 * order they requested it.
 * @author tsa
 * @since 20.11.2016
 */
public class CloseableReentrantLock extends ReentrantLock implements AutoCloseable {
	/**
	 * Constructs a CloseableReentrantLock with the given fairness policy.
	 * <br><br>
	 * If the fairness policy is set to true, threads will acquire the lock in the
	 * order they requested it.
	 *
	 * @param fair the fairness policy of the lock
	 */
	public CloseableReentrantLock(boolean fair) {
		super(fair);
	}

	/**
	 * Constructs a CloseableReentrantLock with the default fairness policy of false.
	 * <br><br>
	 * This means that threads will acquire the lock in the order of their arrival.
	 * <br><br>
	 * <span style="color:red">Note</span> that setting the fairness policy to true may have a significant impact on
	 * performance, as it requires more atomic operations to ensure fairness.
	 */
	public CloseableReentrantLock() {
		this(false);
	}

	/**
	 * Acquires the lock, blocking until it is available.
	 *
	 * @return this instance of CloseableReentrantLock, allowing for method chaining.
	 */
	public CloseableReentrantLock open() {
		this.lock();
		return this;
	}

	/**
	 * Releases the lock. This method is called automatically when the
	 * try-with-resources statement is exited.
	 *
	 * @throws IllegalMonitorStateException if the current thread does not hold the lock.
	 */
	@Override
	public void close() {
		this.unlock();
	}
}