package host.anzo.commons.threading;

import de.mxro.metrics.jre.Metrics;
import delight.async.properties.PropertyNode;
import host.anzo.commons.emergency.metric.Metric;
import host.anzo.commons.emergency.metric.MetricGroupType;
import host.anzo.commons.emergency.metric.MetricResult;
import host.anzo.core.config.EmergencyConfig;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;

import java.util.concurrent.Callable;

/**
 * A wrapper for {@link Callable} tasks that provides error handling, default return value and metrics collection.
 */
@Slf4j
@Metric
@SuppressWarnings("unused")
public class CallableWrapper<V> implements Callable<V> {
	private static final PropertyNode CALLABLE_WRAPPER_METRICS = Metrics.create();
	private static final PropertyNode CALLABLE_WRAPPER_METRICS_VIRTUAL = Metrics.create();

	private final Callable<V> _c;
	private final V _dv;
	private final boolean _v;
	private final String _n;

	/**
	 * Creates a new CallableWrapper for the given Callable task.
	 *
	 * @param c                  the Callable task to wrap
	 * @param defaultReturnValue the default value to return in case of error
	 */
	public CallableWrapper(final Callable<V> c, V defaultReturnValue) {
		this(c, defaultReturnValue, false);
	}

	/**
	 * Creates a new CallableWrapper for the given Callable task.
	 *
	 * @param c                  the Callable task to wrap
	 * @param defaultReturnValue the default value to return in case of error
	 * @param virtual            whether the task should be treated as virtual for metrics collection
	 */
	public CallableWrapper(final Callable<V> c, V defaultReturnValue, final boolean virtual) {
		this(c, defaultReturnValue, virtual, c.getClass().getSimpleName());
	}

	/**
	 * Creates a new CallableWrapper for the given Callable task.
	 *
	 * @param c                  the Callable task to wrap
	 * @param defaultReturnValue the default value to return in case of error
	 * @param name               the name to use for metrics collection
	 */
	public CallableWrapper(final Callable<V> c, V defaultReturnValue, final String name) {
		this(c, defaultReturnValue, false, name);
	}

	/**
	 * Creates a new CallableWrapper for the given Callable task.
	 *
	 * @param c                  the Callable task to wrap
	 * @param defaultReturnValue the default value to return in case of error
	 * @param virtual            whether the task should be treated as virtual for metrics collection
	 * @param name               the name to use for metrics collection
	 */
	public CallableWrapper(final Callable<V> c, V defaultReturnValue, final boolean virtual, final String name) {
		_c = c;
		_dv = defaultReturnValue;
		_v = virtual;
		_n = name != null ? name : c.getClass().getSimpleName();
	}

	/**
	 * Executes the wrapped Callable task.
	 * Returns the result of the wrapped Callable or the default value if an error occurs.
	 * Collects metrics on execution time and errors if metrics are enabled.
	 *
	 * @return the result of the wrapped Callable or the default value
	 * @throws Exception if an error occurs during execution of the wrapped Callable.
	 *                   The original exception thrown by the wrapped Callable is preserved and rethrown.
	 */
	@Override
	public V call() throws Exception {
		V result = _dv;
		if (_c != null) {
			try {
				if (EmergencyConfig.ENABLE_METRICS) {
					final long start = System.nanoTime();
					result = _c.call();
					recordMetric((_v ? CALLABLE_WRAPPER_METRICS_VIRTUAL : CALLABLE_WRAPPER_METRICS), _n + "_time_ns", System.nanoTime() - start);
				} else {
					result = _c.call();
				}
			} catch (final Throwable e) {
				recordMetric((_v ? CALLABLE_WRAPPER_METRICS_VIRTUAL : CALLABLE_WRAPPER_METRICS), _n + "_error");
				log.error("Error while running CallableWrapper:", e);
				throw e;
			}
		}
		return result;
	}

	/**
	 * Returns the metrics for non-virtual CallableWrappers.
	 *
	 * @return the metrics for non-virtual CallableWrappers
	 */
	public static @NotNull MetricResult getMetric() {
		final MetricResult rw = new MetricResult();
		rw.setMetricGroupType(MetricGroupType.THREADPOOL);
		rw.setName("CallableWrapper");
		rw.setData(CALLABLE_WRAPPER_METRICS.render().get());
		return rw;
	}

	/**
	 * Returns the metrics for virtual CallableWrappers.
	 *
	 * @return the metrics for virtual CallableWrappers
	 */
	public static MetricResult getMetricVirtual() {
		final MetricResult rwv = new MetricResult();
		rwv.setMetricGroupType(MetricGroupType.THREADPOOL);
		rwv.setName("CallableWrapperVirtual");
		rwv.setData(CALLABLE_WRAPPER_METRICS_VIRTUAL.render().get());
		return rwv;
	}

	private void recordMetric(PropertyNode metrics, String metricName, long value) {
		if (EmergencyConfig.ENABLE_METRICS) {
			metrics.record(Metrics.value(metricName, value));
		}
	}

	private void recordMetric(PropertyNode metrics, String metricName) {
		if (EmergencyConfig.ENABLE_METRICS) {
			metrics.record(Metrics.happened(metricName));
		}
	}
}