package host.anzo.commons.utils;

import host.anzo.commons.model.RouletteRandomEntry;
import host.anzo.commons.text.RandomNameGenerator;
import org.apache.commons.lang3.RandomStringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

/**
 * Utility class for generating random numbers and strings.
 * Provides various methods to generate random values of different types.
 * @author ANZO
 */
public class Rnd {
	/**
	 * Gets a random number from 0 (inclusive) to 1 (exclusive).
	 *
	 * @return a random double value between 0.0 and 1.0
	 */
	public static double get() {
		return ThreadLocalRandom.current().nextDouble();
	}

	/**
	 * Gets a random integer between 0 (inclusive) and n (exclusive).
	 *
	 * @param n The superior limit (exclusive)
	 * @return if positive: A number from 0 to n-1, if negative A number from n-1 to 0
	 */
	public static int get(int n) {
		if (n == 0) {
			return 0;
		}
		final int sign = n > 0 ? 1 : -1;
		return sign * (ThreadLocalRandom.current().nextInt(Math.abs(n)));
	}

	/**
	 * Gets a random long number between 0 (inclusive) and n (exclusive).
	 *
	 * @param n The superior limit (exclusive)
	 * @return if positive: A number from 0 to n-1, if negative A number from n-1 to 0
	 */
	public static long get(long n) {
		if (n == 0) {
			return 0;
		}
		final int sign = n > 0 ? 1 : -1;
		return sign * (long) (ThreadLocalRandom.current().nextDouble() * n);
	}

	/**
	 * Gets a random double number between 0 (inclusive) and n (exclusive).
	 *
	 * @param n The superior limit (exclusive)
	 * @return if positive: A number from 0 to n-1, if negative A number from n-1 to 0
	 */
	public static double get(double n) {
		if (n == 0) {
			return 0;
		}
		final int sign = n > 0 ? 1 : -1;
		return sign * (ThreadLocalRandom.current().nextDouble() * n);
	}

	/**
	 * Gets a random float number between 0 (inclusive) and n (exclusive).
	 *
	 * @param n The superior limit (exclusive)
	 * @return if positive: A number from 0 to n-1, if negative A number from n-1 to 0
	 */
	public static float get(float n) {
		if (n == 0) {
			return 0;
		}
		final int sign = n > 0 ? 1 : -1;
		return sign * (ThreadLocalRandom.current().nextFloat() * n);
	}

	/**
	 * Gets a random integer between min (inclusive) and max (inclusive).
	 *
	 * @param min The lower limit (inclusive)
	 * @param max The upper limit (inclusive)
	 * @return a random integer between min and max
	 */
	public static int get(int min, int max) {
		return min + get(max - min + 1);
	}

	/**
	 * Gets a random long number between min (inclusive) and max (inclusive).
	 *
	 * @param min The lower limit (inclusive)
	 * @param max The upper limit (inclusive)
	 * @return a random long number between min and max
	 */
	public static long get(long min, long max) {
		return min + get(max - min + 1);
	}

	/**
	 * Gets a random float number between min (inclusive) and max (exclusive).
	 *
	 * @param min The lower limit (inclusive)
	 * @param max The upper limit (exclusive)
	 * @return a random float number between min and max
	 */
	public static float get(float min, float max) {
		return min + get(max - min);
	}

	/**
	 * Gets a random double number between min (inclusive) and max (exclusive).
	 *
	 * @param min The lower limit (inclusive)
	 * @param max The upper limit (exclusive)
	 * @return a random double number between min and max
	 */
	public static double get(double min, double max) {
		return min + get(max - min);
	}

	/**
	 * Gets a random integer from the full range of int values.
	 *
	 * @return a random integer
	 */
	public static int nextInt() {
		return ThreadLocalRandom.current().nextInt();
	}

	/**
	 * Gets a random double from the full range of double values.
	 *
	 * @return a random double
	 */
	public static double nextDouble() {
		return ThreadLocalRandom.current().nextDouble();
	}

	/**
	 * Gets a random number from a Gaussian distribution.
	 *
	 * @return a random double from a Gaussian distribution
	 */
	public static double nextGaussian() {
		return ThreadLocalRandom.current().nextGaussian();
	}

	/**
	 * Gets a random boolean value.
	 *
	 * @return a random boolean
	 */
	public static boolean nextBoolean() {
		return ThreadLocalRandom.current().nextBoolean();
	}

	/**
	 * Fills the provided byte array with random bytes.
	 *
	 * @param bytes the byte array to fill
	 * @return the filled byte array
	 */
	public static byte[] nextBytes(final byte[] bytes) {
		ThreadLocalRandom.current().nextBytes(bytes);
		return bytes;
	}

	/**
	 * Randomizer for chance calculation.
	 * Recommending to use instead Rnd.get(n, n).
	 *
	 * @param chance in percent from 0 to 100
	 * @return {@code true} if success
	 * If chance &lt;= 0, return false
	 * If chance &gt;= 100, return true
	 */
	public static boolean getChance(int chance) {
		return chance >= 1 && (chance > 99 || ThreadLocalRandom.current().nextInt(99) + 1 <= chance);
	}

	/**
	 * Randomizer for chance calculation.
	 * Recommending to use instead Rnd.get(n, n) if we need high precision values.
	 *
	 * @param chance in percent from 0 to 100
	 * @return {@code true} if success
	 * If chance &lt;= 0, return false
	 * If chance &gt;= 100, return true
	 */
	public static boolean getChance(double chance) {
		return ThreadLocalRandom.current().nextDouble() * 100. <= chance;
	}

	/**
	 * Randomizer for chance calculation with a divider.
	 *
	 * @param chance in percent from 0 to 100
	 * @param divider the divider to adjust the chance
	 * @return {@code true} if success
	 */
	public static boolean getChance(int chance, int divider) {
		return getChance((double)chance / (double) divider);
	}

	/**
	 * Gets a random element from the provided array.
	 *
	 * @param list the array to select from
	 * @param <E> the type of elements in the array
	 * @return a random element from the array
	 */
	public static <E> E get(E @NotNull [] list) {
		return list[get(list.length)];
	}

	/**
	 * Gets a random integer from the provided integer array.
	 *
	 * @param list the integer array to select from
	 * @return a random integer from the array
	 */
	public static int get(int @NotNull [] list) {
		return list[get(list.length)];
	}

	/**
	 * Gets a random element from the provided list.
	 *
	 * @param list the list to select from
	 * @param <E> the type of elements in the list
	 * @return a random element from the list, or null if the list is empty
	 */
	public static <E> @Nullable E get(@NotNull List<E> list) {
		if (list.isEmpty()) {
			return null;
		}
		return list.get(get(list.size()));
	}

	/**
	 * Gets a random element from the provided list or returns a default value if the list is empty.
	 *
	 * @param list the list to select from
	 * @param defaultValue the value to return if the list is empty
	 * @param <E> the type of elements in the list
	 * @return a random element from the list or the default value
	 */
	public static <E> E get(@NotNull List<E> list, E defaultValue) {
		if (list.isEmpty()) {
			return defaultValue;
		}
		return list.get(get(list.size()));
	}

	/**
	 * Gets a random element from the provided collection.
	 *
	 * @param collection the collection to select from
	 * @param <E> the type of elements in the collection
	 * @return a random element from the collection, or null if the collection is empty
	 */
	@SuppressWarnings("unchecked")
	public static <E> @Nullable E get(@NotNull Collection<E> collection) {
		if (collection.isEmpty()) {
			return null;
		}
		return (E) collection.toArray()[get(collection.size())];
	}

	/**
	 * Gets a random element from the provided enum class.
	 *
	 * @param clazz the class of the enum to select from
	 * @param <E> the type of the enum
	 * @return a random enum constant from the specified class
	 */
	public static <E extends Enum<E>> E get(@NotNull Class<E> clazz) {
		return clazz.getEnumConstants()[Rnd.get(clazz.getEnumConstants().length)];
	}

	/**
	 * Gets a list of random entries from a roulette selection.
	 *
	 * @param input the list of entries to select from
	 * @param count the number of entries to select
	 * @return a list of randomly selected RouletteRandomEntry objects
	 */
	public static @NotNull List<RouletteRandomEntry> getRouletteEntries(List<RouletteRandomEntry> input, int count) {
		final List<RouletteRandomEntry> selectedMainTemplates = new ArrayList<>();
		final List<RouletteRandomEntry> copyItems = new ArrayList<>(input);

		for (int i = 0; i < count; i++) {
			final int totalChance = copyItems.stream().mapToInt(RouletteRandomEntry::chance).sum();
			final int randomValue = Rnd.get(totalChance);

			int accumulatedChance = 0;
			int selectedIndex = -1;

			for (int j = 0; j < copyItems.size(); j++) {
				accumulatedChance += copyItems.get(j).chance();
				if (randomValue < accumulatedChance) {
					selectedIndex = j;
					break;
				}
			}

			if (selectedIndex != -1) {
				selectedMainTemplates.add(copyItems.remove(selectedIndex));
			}
		}
		return selectedMainTemplates;
	}

	/**
	 * Gets a single random entry from a roulette selection.
	 *
	 * @param input the list of entries to select from
	 * @return a randomly selected RouletteRandomEntry object, or null if the input list is empty
	 */
	public static @Nullable RouletteRandomEntry getRouletteEntry(List<RouletteRandomEntry> input) {
		return getRouletteEntries(input, 1).stream().findFirst().orElse(null);
	}

	/**
	 * Generates a random number within a specified range, influenced by server tick and random seed.
	 *
	 * @param min the minimum value (inclusive)
	 * @param max the maximum value (exclusive)
	 * @param serverTick the server tick to influence randomness
	 * @param randomSeed the seed to influence randomness
	 * @return a random double value between min and max
	 */
	public static double getRandomWithTick(double min, double max, long serverTick, long randomSeed) {
		final long randomizeSeed = ((serverTick + randomSeed) * 0x01234567 * 0x89ABCDEF * 0x01234567 * 0x89ABCDEF * 'R' * 'a' * 'n' * 'd' * 'o' * 'm' * (randomSeed + 0x89ABCDEF)) & 0xffffffffL;
		final long FIXED_RANDOM = randomizeSeed + (((randomizeSeed) % 11 - (randomizeSeed) % 10 + (randomizeSeed) % 9 - (randomizeSeed) % 8 + (randomizeSeed) % 7 - (randomizeSeed) % 6 + (randomizeSeed) % 5 - (randomizeSeed) % 4 + (randomizeSeed) % 3 - (randomizeSeed) % 2 + (randomizeSeed))) & 0xffffffffL;
		return FIXED_RANDOM % (10000) / (10000.0f - 1.0f) * (max - min) + min;
	}

	/**
	 * Generates a random string of alphabetic characters of a specified length.
	 *
	 * @param length the length of the random string to generate
	 * @return a random alphabetic string of the specified length
	 */
	public static @NotNull String getString(int length) {
		return RandomStringUtils.secure().nextAlphanumeric(length);
	}

	/**
	 * Generates a random string of hexadecimal characters of a specified length.
	 *
	 * @param length the length of the random hexadecimal string to generate
	 * @return a random hexadecimal string of the specified length
	 */
	public static @NotNull String getStringHex(int length) {
		return RandomStringUtils.secure().next(length, "0123456789abcdef");
	}

	/**
	 * Retrieves a random human-readable
	 * @return a randomly generated name
	 */
	public static @NotNull String getName() {
		return RandomNameGenerator.getInstance().nextName();
	}
}