package host.anzo.commons.concurrent;

import java.util.concurrent.locks.ReentrantLock;

/**
 * A CloseableReentrantLock is a specialized implementation of a ReentrantLock
 * that implements the AutoCloseable interface. This allows the lock to be used
 * in a try-with-resources statement, ensuring that the lock is properly released
 * when it is no longer needed.
 * <br><br>
 * Unlike the standard {@link ReentrantLock}, this implementation defaults to a
 * <b>fair</b> locking policy (FIFO order). This means threads will acquire the lock
 * in the order they requested it, but it may impact performance due to additional
 * synchronization overhead.
 * <br><br>
 * <span style="color:red">Note:</span> Fair locks are generally slower than non-fair locks,
 * as they require more atomic operations to maintain strict ordering.
 *
 * @author tsa, ANZO
 * @since 20.11.2016
 */
public class CloseableReentrantLock extends ReentrantLock implements AutoCloseable {
	/**
	 * Constructs a CloseableReentrantLock with the given fairness policy.
	 * <br><br>
	 * If the fairness policy is set to {@code true}, threads will acquire the lock
	 * in the order they requested it (FIFO). If {@code false}, the lock permits
	 * barging, allowing threads to acquire the lock out of order for better performance.
	 * <br><br>
	 * <span style="color:red">Warning:</span> Fair locks may significantly reduce throughput
	 * under high contention.
	 *
	 * @param fair {@code true} for a fair lock, {@code false} for a non-fair lock
	 */
	public CloseableReentrantLock(boolean fair) {
		super(fair);
	}

	/**
	 * Constructs a CloseableReentrantLock with a <b>fair</b> locking policy (FIFO order).
	 * <br><br>
	 * This is a deviation from the standard {@link ReentrantLock}, which defaults to non-fair mode.
	 */
	public CloseableReentrantLock() {
		this(true);
	}

	/**
	 * Acquires the lock, blocking until it is available.
	 * @return this instance of CloseableReentrantLock, allowing for method chaining.
	 */
	public CloseableReentrantLock open() {
		this.lock();
		return this;
	}

	/**
	 * Releases the lock. This method is called automatically when the
	 * try-with-resources block is exited.
	 * <br><br>
	 * If the current thread does not hold the lock, an {@link IllegalMonitorStateException} is thrown.
	 * This method is idempotent; calling it multiple times has no additional effect if the thread
	 * still holds the lock.
	 *
	 * @throws IllegalMonitorStateException if the current thread does not hold the lock.
	 */
	@Override
	public void close() {
		this.unlock();
	}
}