package host.anzo.commons.emergency.memory.watchers;

import lombok.extern.slf4j.Slf4j;

import javax.management.NotificationEmitter;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.MemoryType;

/**
 * The G1Watcher class implements the IMemoryWatcher interface and is responsible
 * for monitoring the memory usage of G1 garbage collector memory pools. It registers
 * a notification listener to receive updates when the memory usage exceeds a specified
 * threshold.
 *
 * @author ANZO
 * @since 08.07.2017
 */
@Slf4j
public class G1Watcher implements IMemoryWatcher {
	public G1Watcher() {
	}

	/**
	 * Registers the specified MemoryPoolMXBean for monitoring. It sets the usage
	 * threshold to 85% of the maximum usage and adds a notification listener to
	 * listen for memory usage notifications.
	 *
	 * @param pool the MemoryPoolMXBean to register for monitoring
	 */
	@Override
	public void register(MemoryPoolMXBean pool) {
		long max = pool.getUsage().getMax();
		pool.setUsageThreshold(Math.round(max * 0.85));
		
		NotificationEmitter emitter = (NotificationEmitter) ManagementFactory.getMemoryMXBean();
		emitter.addNotificationListener(new PoolListener(pool), null, null);
		log.info("Listening {}", pool.getName());
	}

	/**
	 * Validates whether the specified MemoryPoolMXBean is suitable for monitoring.
	 * A pool is considered valid if it is of type HEAP, supports usage thresholds,
	 * supports collection usage thresholds, and its name starts with "G1".
	 *
	 * @param pool the MemoryPoolMXBean to validate
	 * @return true if the pool is valid for monitoring; false otherwise
	 */
	@Override
	public boolean isValid(MemoryPoolMXBean pool) {
		return pool.getType() == MemoryType.HEAP && pool.isUsageThresholdSupported() && pool.isCollectionUsageThresholdSupported()
				&& pool.getName().startsWith("G1");
	}
}