package host.anzo.core.service;

import host.anzo.commons.annotations.startup.StartupComponent;
import host.anzo.commons.interfaces.startup.IShutdownable;
import lombok.Getter;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;

import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * A singleton service for monitoring file system changes in registered directories.
 * Supports recursive directory monitoring with file extension filtering and asynchronous event notifications.
 *
 * <p>Example usage:
 * <pre>{@code
 * // Register a listener for JSON files in a directory
 * FileWatchListener listener = new FileWatchListener() {
 *      void onFileChanged(FileWatchEvent event) {
 *          void onFileChanged(FileWatchEvent event) {
 *          }
 *      }
 * }
 *
 * FileWatcherService.getInstance().registerDirectory(
 *     Paths.get("/path/to/dir"),
 *     listener,
 *     ".json"
 * );
 *
 * // Unregister listener
 * FileWatcherService.getInstance().unregisterListener(listener);
 *
 * // Shutdown the service when no longer needed
 * FileWatcherService.getInstance().shutdown();
 * }
 * </pre>
 *
 * @author ANZO
 * @since 05.04.2025
 */
@Slf4j
@StartupComponent("Service")
public class FileWatcherService implements IShutdownable {
	@Getter(lazy = true)
	private static final FileWatcherService instance = new FileWatcherService();

	private final WatchService watchService;
	private final Map<WatchKey, WatchedDirectory> watchKeys;
	private volatile boolean running;

	private final Map<Path, Long> lastModifiedMap = new ConcurrentHashMap<>();

	/**
	 * Private constructor to enforce singleton pattern.
	 * Initializes the file monitoring infrastructure and starts the background monitoring thread.
	 */
	@SneakyThrows
	private FileWatcherService(){
		this.watchService = FileSystems.getDefault().newWatchService();
		this.watchKeys = new ConcurrentHashMap<>();
		this.running = true;
		startMonitoring();
	}

	/**
	 * Registers a directory for monitoring with optional file extension filtering.
	 * Recursively registers all subdirectories found at the time of registration.
	 *
	 * @param directory The directory path to monitor
	 * @param listener  The listener to receive file change events
	 * @param extensions Varargs of file extensions to filter (e.g., "txt", "log")
	 */
	public void registerDirectory(Path directory, FileWatchListener listener, String... extensions) {
		registerDirectory(directory, listener, Arrays.asList(extensions));
	}

	/**
	 * Registers a directory for monitoring with extension filtering.
	 * This overload accepts a list of extensions instead of varargs.
	 *
	 * @param directory   The directory path to monitor
	 * @param listener    The listener to receive file change events
	 * @param extensions  List of file extensions to filter
	 */
	public void registerDirectory(Path directory, FileWatchListener listener, List<String> extensions) {
		try {
			final WatchedDirectory watchedDir = new WatchedDirectory(directory, listener, extensions);
			registerDirectoryRecursive(watchedDir);

			Files.walkFileTree(directory, new SimpleFileVisitor<>() {
				@Override
				public @NotNull FileVisitResult preVisitDirectory(@NotNull Path dir, @NotNull BasicFileAttributes attrs) throws IOException {
					registerDirectoryRecursive(new WatchedDirectory(dir, listener, extensions));
					return FileVisitResult.CONTINUE;
				}
			});
		}
		catch (Exception e) {
			log.error("Error registering directory {}", directory, e);
		}
	}

	/**
	 * Recursively registers a directory with the watch service.
	 * This method ensures the path exists and is a valid directory before registering it
	 * with the watch service to monitor create/delete/modify events.
	 *
	 * @param watchedDir The directory to register along with its associated listener and extensions
	 * @throws IOException if registration fails or path is not a directory
	 */
	private void registerDirectoryRecursive(@NotNull WatchedDirectory watchedDir) throws IOException {
		final Path directory = watchedDir.path();
		if (!Files.isDirectory(directory)) {
			throw new IllegalArgumentException("Path must be a directory: " + directory);
		}

		final WatchKey key = directory.register(watchService,
				StandardWatchEventKinds.ENTRY_CREATE,
				StandardWatchEventKinds.ENTRY_DELETE,
				StandardWatchEventKinds.ENTRY_MODIFY);

		watchKeys.put(key, watchedDir);
	}

	/**
	 * Unregisters all directories associated with the specified listener.
	 * Stops receiving events for all directories monitored by this listener.
	 *
	 * @param listener The listener to unregister
	 */
	public void unregisterListener(FileWatchListener listener) {
		watchKeys.values().removeIf(watchedDir -> watchedDir.listener() == listener);
	}

	/**
	 * Starts the background monitoring thread that watches for file system events.
	 * This method initializes an asynchronous task that continuously polls for WatchKeys,
	 * processes triggered events, and manages key resets to continue monitoring.
	 */
	private void startMonitoring() {
		ThreadPoolService.getInstance().submit(() -> {
			while (running) {
				try {
					final WatchKey key = watchService.take();
					final WatchedDirectory watchedDir = watchKeys.get(key);

					if (watchedDir == null) {
						continue;
					}

					for (WatchEvent<?> event : key.pollEvents()) {
						handleEvent(watchedDir, event);
					}

					if (!key.reset()) {
						watchKeys.remove(key);
					}
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					break;
				} catch (ClosedWatchServiceException e) {
					break;
				}
			}
		});
	}

	/**
	 * Handles individual file system events.
	 * Processes directory creation/deletion/modify events and notifies the appropriate listener.
	 *
	 * @param watchedDir The watched directory that triggered the event
	 * @param event      The file system event to process
	 */
	private void handleEvent(@NotNull WatchedDirectory watchedDir, @NotNull WatchEvent<?> event) {
		final Path dir = watchedDir.path();
		final Path fullPath = dir.resolve((Path) event.context());
		final FileWatchListener listener = watchedDir.listener();

		if (!matchesExtension(fullPath, watchedDir.extensions())) {
			return;
		}

		final WatchEvent.Kind<?> kind = event.kind();

		if (kind == StandardWatchEventKinds.ENTRY_CREATE) {
			if (Files.isDirectory(fullPath)) {
				try {
					registerDirectoryRecursive(new WatchedDirectory(fullPath, listener, watchedDir.extensions()));
				} catch (IOException e) {
					notifyListener(listener, FileWatchEvent.error(e));
				}
			}
			notifyListener(listener, FileWatchEvent.created(fullPath));
		} else if (kind == StandardWatchEventKinds.ENTRY_DELETE) {
			lastModifiedMap.remove(fullPath);
			notifyListener(listener, FileWatchEvent.deleted(fullPath));
		} else if (kind == StandardWatchEventKinds.ENTRY_MODIFY) {
			final long now = System.currentTimeMillis();
			final Long lastEventTime = lastModifiedMap.get(fullPath);
			if (lastEventTime != null && (now - lastEventTime) < 1000)
				return;
			lastModifiedMap.put(fullPath, now);
			notifyListener(listener, FileWatchEvent.modified(fullPath));
		} else if (kind == StandardWatchEventKinds.OVERFLOW) {
			notifyListener(listener, FileWatchEvent.overflow());
		}
	}

	/**
	 * Checks if a file matches any of the registered extensions.
	 * Returns true when no extensions are specified (no filtering).
	 *
	 * @param file       The file to check
	 * @param extensions The list of extensions to match against
	 * @return true if the file matches any extension or no extensions are specified
	 */
	private boolean matchesExtension(Path file, List<String> extensions) {
		if (extensions == null || extensions.isEmpty()) {
			return true;
		}

		final String fileName = file.getFileName().toString();
		return extensions.stream().anyMatch(fileName::endsWith);
	}

	/**
	 * Asynchronously notifies a listener about a file system event.
	 * Events are delivered through the service's thread pool.
	 *
	 * @param listener The listener to notify
	 * @param event    The event to deliver
	 */
	private void notifyListener(FileWatchListener listener, FileWatchEvent event) {
		ThreadPoolService.getInstance().submit(() -> {
			try {
				listener.onFileChanged(event);
			} catch (Exception e) {
				listener.onFileChanged(FileWatchEvent.error(e));
			}
		});
	}

	/**
	 * Initiates an orderly shutdown of the file watcher service.
	 */
	@Override
	public void onShutdown() {
		running = false;
		try {
			watchService.close();
		} catch (IOException ignored) {
		}
		watchKeys.clear();
	}

	/**
	 * A listener interface for receiving file system change events.
	 */
	public interface FileWatchListener {
		/**
		 * Called when a file system change event occurs.
		 * This method is executed asynchronously in a thread pool.
		 *
		 * @param event The event describing the file system change
		 */
		void onFileChanged(@NotNull FileWatchEvent event);
	}

	/**
	 * Represents a file system change event.
	 * Provides factory methods for creating common event types.
	 */
	public @Getter static class FileWatchEvent {
		public enum FileWatchEventType { CREATED, DELETED, MODIFIED, ERROR, OVERFLOW }

		private final FileWatchEventType type;
		private final Path path;
		private final Exception error;

		private FileWatchEvent(FileWatchEventType type, Path path, Exception error) {
			this.type = type;
			this.path = path;
			this.error = error;
		}

		/**
		 * Creates a CREATED event for a file at the specified path.
		 *
		 * @param path The path of the created file
		 * @return A new FileWatchEvent with Type.CREATED
		 */
		@Contract(value = "_ -> new", pure = true)
		public static @NotNull FileWatcherService.FileWatchEvent created(Path path) {
			return new FileWatchEvent(FileWatchEventType.CREATED, path, null);
		}

		/**
		 * Creates a DELETED event for a file at the specified path.
		 *
		 * @param path The path of the deleted file
		 * @return A new FileWatchEvent with Type.DELETED
		 */
		@Contract(value = "_ -> new", pure = true)
		public static @NotNull FileWatcherService.FileWatchEvent deleted(Path path) {
			return new FileWatchEvent(FileWatchEventType.DELETED, path, null);
		}

		/**
		 * Creates a MODIFIED event for a file at the specified path.
		 *
		 * @param path The path of the modified file
		 * @return A new FileWatchEvent with Type.MODIFIED
		 */
		@Contract(value = "_ -> new", pure = true)
		public static @NotNull FileWatcherService.FileWatchEvent modified(Path path) {
			return new FileWatchEvent(FileWatchEventType.MODIFIED, path, null);
		}

		/**
		 * Creates an ERROR event containing the exception that occurred.
		 *
		 * @param e The exception that caused the error
		 * @return A new FileWatchEvent with Type.ERROR
		 */
		@Contract(value = "_ -> new", pure = true)
		public static @NotNull FileWatcherService.FileWatchEvent error(Exception e) {
			return new FileWatchEvent(FileWatchEventType.ERROR, null, e);
		}

		/**
		 * Creates an OVERFLOW event indicating that events were lost or discarded.
		 *
		 * @return A new FileWatchEvent with Type.OVERFLOW
		 */
		@Contract(value = " -> new", pure = true)
		public static @NotNull FileWatcherService.FileWatchEvent overflow() {
			return new FileWatchEvent(FileWatchEventType.OVERFLOW, null, null);
		}
	}

	/**
	 * Immutable record representing a directory being watched.
	 * @param path the directory path
	 * @param listener the associated event listener
	 * @param extensions the file extensions to monitor
	 */
	private record WatchedDirectory(Path path, FileWatchListener listener, List<String> extensions) {
		private WatchedDirectory(Path path, FileWatchListener listener, List<String> extensions) {
			this.path = path;
			this.listener = listener;
			this.extensions = extensions != null ?
					extensions.stream()
							.map(ext -> ext.startsWith(".") ? ext : "." + ext)
							.collect(Collectors.toList()) :
					Collections.emptyList();
		}
	}
}