package host.anzo.commons.collection;

import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * A {@link LinkedHashMap} implementation with index-based access to keys and values.
 * Maintains insertion order like {@code LinkedHashMap} while allowing retrieval
 * of elements by index via {@code getKeyAt(index)} and {@code getValueAt(index)}.
 *
 * @param <K> the type of keys maintained by this map
 * @param <V> the type of mapped values
 *
 * @author ANZO
 * @since 5/16/2024
 */
public class IndexableHashMap<K, V> extends LinkedHashMap<K, V> {
	/**
	 * Internal list to track keys in insertion order.
	 */
	private final List<K> keyList = new ArrayList<>();

	/**
	 * Associates the specified value with the specified key in this map.
	 * If the key is not already present in the map, it is added to the key list.
	 *
	 * @param key   the key with which the specified value is to be associated (non-null)
	 * @param value the value to be associated with the specified key (non-null)
	 * @return the previous value associated with the key, or {@code null} if none
	 * @throws NullPointerException if the key or value is null
	 */
	@Override
	public V put(@NotNull K key, @NotNull V value) {
		if (!keyList.contains(key)) {
			keyList.add(key);
		}
		return super.put(key, value);
	}

	/**
	 * Copies all entries from the specified map to this map.
	 * Ensures keys are added to the key list if not already present.
	 *
	 * @param m the map whose entries are to be added to this map
	 */
	@Override
	public void putAll(Map<? extends K, ? extends V> m) {
		for (Map.Entry<? extends K, ? extends V> entry : m.entrySet()) {
			put(entry.getKey(), entry.getValue());
		}
	}

	/**
	 * Clears all entries from this map and the internal key list.
	 */
	@Override
	public void clear() {
		keyList.clear();
		super.clear();
	}

	/**
	 * Returns a copy of the internal key list to prevent external modification.
	 * The list reflects keys in insertion order.
	 *
	 * @return a new {@link List} containing the keys in insertion order
	 */
	public List<K> getKeys() {
		return new ArrayList<>(keyList);
	}

	/**
	 * Returns the index of the specified key in the insertion order list.
	 *
	 * @param key the key to search for
	 * @return the index of the key, or {@code -1} if not found
	 */
	public int getKeyIndex(K key) {
		return keyList.indexOf(key);
	}

	/**
	 * Retrieves the key at the specified index in the insertion order list.
	 *
	 * @param index the index of the key to retrieve
	 * @return the key at the specified index, or {@code null} if the index is out of bounds
	 */
	public K getKeyAt(int index) {
		if (keyList.size() > index && index >= 0) {
			return keyList.get(index);
		}
		return null;
	}

	/**
	 * Retrieves the value at the specified index in the insertion order list.
	 *
	 * @param index the index of the value to retrieve
	 * @return the value associated with the key at the specified index,
	 *         or {@code null} if the index is invalid or the key has no mapping
	 */
	public V getValueAt(int index) {
		K key = getKeyAt(index);
		if (key != null) {
			return get(key);
		}
		return null;
	}

	/**
	 * Removes the mapping for the specified key and updates the internal key list.
	 * <p>
	 * <b>Performance Note:</b> This method rebuilds the key list, which may be
	 * inefficient for frequent remove operations.
	 *
	 * @param key the key to remove (non-null)
	 * @return the previous value associated with the key, or {@code null} if none
	 */
	@Override
	public V remove(@NotNull Object key) {
		V removedEntry = super.remove(key);
		keyList.clear();
		keyList.addAll(super.keySet()); // Ensures the list reflects current map state
		return removedEntry;
	}
}