package host.anzo.commons.emergency.metric;

import host.anzo.classindex.ClassIndex;
import host.anzo.commons.annotations.startup.StartupComponent;
import host.anzo.commons.utils.ClassUtils;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.text.TextStringBuilder;

import java.lang.reflect.Modifier;
import java.util.*;

/**
 * Service for collecting and managing diagnostic metrics from various components.
 * <p>
 * This service automatically discovers and registers all metric implementations annotated with {@link Metric}
 * during initialization. Metrics can be organized into groups and retrieved in a structured JSON format.
 * </p>
 *
 * <p>The service follows the singleton pattern and is initialized during application startup
 * (marked with {@link StartupComponent}).</p>
 *
 * @author ANZO
 * @since 05.04.2021
 * @see Metric
 * @see IMetric
 * @see MetricResult
 * @see MetricGroupType
 */

@Slf4j
@StartupComponent("Diagnostic")
public class MetricService {
	@Getter(lazy=true)
	private final static MetricService instance = new MetricService();

	private final Set<IMetric> metricInstances = new HashSet<>();

	private MetricService() {
		final Set<Class<?>> parentMetrics = new HashSet<>();
		ClassIndex.getAnnotated(Metric.class).forEach(clazz -> {
			if (!IMetric.class.isAssignableFrom(clazz) || Modifier.isAbstract(clazz.getModifiers())) {
				return;
			}

			final Metric metricAnnotation = clazz.getAnnotation(Metric.class);
			if (metricAnnotation.fromParent() && parentMetrics.contains(clazz.getSuperclass())) {
				return;
			}

			if (metricAnnotation.fromParent()) {
				parentMetrics.add(clazz.getSuperclass());
			}

			final Object instance = ClassUtils.singletonInstance(clazz);
			if (instance instanceof IMetric metricInstance) {
				metricInstances.add(metricInstance);
			}
		});
	}

	/**
	 * Collects and formats all registered metrics into a JSON string.
	 * <p>
	 * Example output:
	 * <pre>{@code
	 * {
	 *     "version": "1.0.0",
	 *     "metricGroup1": {
	 *         "metric1": "value1",
	 *         "metric2": "value2"
	 *     },
	 *     "metricGroup2": {
	 *         "metric3": "value3",
	 *         "metric4": "value4"
	 *     }
	 * }
	 * }</pre>
	 *
	 * @return JSON formatted string containing all collected metrics
	 * @see IMetric#getMetric()
	 * @see MetricResult
	 */
	public String getMetrics() {
		final TextStringBuilder allBuilder = new TextStringBuilder("{").appendln("\"version\": \"1.0.0\",");
		final Map<MetricGroupType, List<String>> metricsResult = new HashMap<>();
		for (IMetric metricInstance : metricInstances) {
			final List<MetricResult> metrics = metricInstance.getMetric();
			for (MetricResult metric : metrics) {
				metricsResult.computeIfAbsent(metric.getMetricGroupType(), k -> new ArrayList<>()).add("\"" + metric.getName() + "\" : " + metric.getData());
			}
		}

		final List<String> metricGroupResults = new ArrayList<>();
		for (Map.Entry<MetricGroupType, List<String>> entry : metricsResult.entrySet()) {
			metricGroupResults.add("\"" + entry.getKey() + "\" : {" + String.join(",", entry.getValue()) +"}");
		}
		allBuilder.appendln(String.join(",", metricGroupResults));
		allBuilder.appendln("}");
		return allBuilder.get();
	}
}