package host.anzo.core.webserver.model;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.core.JsonProcessingException;
import host.anzo.commons.config.ConfigTypeCaster;
import host.anzo.commons.utils.JsonUtils;
import host.anzo.core.service.HttpService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.text.StringEscapeUtils;
import spark.Request;

import java.util.*;

/**
 * @author ANZO
 * @since 04.06.2017
 */
@Slf4j
public abstract class AWebRequest {
	@JsonIgnore
	private Map<String, String[]> queryMap;

	@JsonIgnore
	private final Map<String, Map<String, String[]>> maps = new HashMap<>();

	@JsonIgnore
	private final Map<String, List<String>> lists = new HashMap<>();

	@JsonIgnore
	private final Map<String, String> params = new HashMap<>();

	protected AWebRequest() {
	}

	public AWebRequest(Request request) {
		if (request != null) {
			params.putAll(request.params());
			queryMap = request.queryMap().toMap();
			for (Map.Entry<String, String[]> entry : queryMap.entrySet()) {
				if (entry.getValue() == null && request.queryMap(entry.getKey()) != null) {
					maps.put(entry.getKey(), request.queryMap(entry.getKey()).toMap());
				}
			}
			for (Map.Entry<String, String[]> entry : request.raw().getParameterMap().entrySet()) {
				if (!maps.containsKey(entry.getKey()) && entry.getKey().contains("[]")) {
					final String key = entry.getKey().replace("[]", "");
					for (String value : entry.getValue()) {
						lists.computeIfAbsent(key, k -> new ArrayList<>()).add(value);
					}
				}
			}
		}
	}

	protected String getString(String paramName) {
		return getString(paramName, "");
	}

	protected String getString(String paramName, String defaultValue) {
		if (queryMap != null && queryMap.containsKey(paramName) && queryMap.get(paramName) != null) {
			return HttpService.getInstance().stripXSS(queryMap.get(paramName)[0]).replaceAll("\\.\\./", "");
		}
		return defaultValue;
	}

	protected long getLong(String paramName) {
		if (queryMap != null
		    && queryMap.containsKey(paramName)
		    && queryMap.get(paramName) != null
		    && StringUtils.isNoneEmpty(queryMap.get(paramName))) {
			try {
				return Long.parseLong(queryMap.get(paramName)[0].trim());
			}
			catch (Exception e) {
				return 0;
			}
		}
		return 0;
	}

	protected int getInteger(String paramName) {
		return getInteger(paramName, 0);
	}

	protected int getInteger(String paramName, int defaultValue) {
		if (queryMap != null
		    && queryMap.containsKey(paramName)
		    && queryMap.get(paramName) != null
		    && StringUtils.isNoneEmpty(queryMap.get(paramName))) {
			try {
				return Integer.parseInt(queryMap.get(paramName)[0].trim());
			}
			catch (Exception e) {
				return defaultValue;
			}
		}
		return defaultValue;
	}

	protected double getDouble(String paramName) {
		return getDouble(paramName, 0);
	}

	protected double getDouble(String paramName, double defaultValue) {
		if (queryMap != null
				&& queryMap.containsKey(paramName)
				&& queryMap.get(paramName) != null
				&& StringUtils.isNoneEmpty(queryMap.get(paramName))) {
			try {
				return Double.parseDouble(queryMap.get(paramName)[0].trim());
			}
			catch (Exception e) {
				return defaultValue;
			}
		}
		return defaultValue;
	}

	protected boolean getBoolean(String paramName) {
		if (queryMap != null && queryMap.containsKey(paramName) && queryMap.get(paramName) != null) {
			try {
				return Boolean.parseBoolean(queryMap.get(paramName)[0].toLowerCase().trim());
			}
			catch (Exception e) {
				return false;
			}
		}
		return false;
	}

	protected <T extends Enum<T>> T getEnum(String paramName, Class<T> enumClass, T defaultValue) {
		final int ordinalValue = getInteger(paramName);
		try {
			return enumClass.getEnumConstants()[ordinalValue];
		}
		catch (Exception e) {
			return defaultValue;
		}
	}

	protected Map<String,String> getMap(String paramName) {
		final Map<String, String> map = new HashMap<>();
		if (maps.containsKey(paramName) && maps.get(paramName) != null) {
			for (Map.Entry<String, String[]> entry : maps.get(paramName).entrySet()) {
				map.put(entry.getKey(), entry.getValue()[0]);
			}
		}
		return map;
	}

	protected <T> List<T> getList(Class<T> type, String paramName) {
		final List<String> stringList = lists.get(paramName);
		if (stringList == null) {
			return Collections.emptyList();
		}
		final List<T> result = new ArrayList<>();
		for (String stringValue : stringList) {
			try {
				result.add(ConfigTypeCaster.cast(type, stringValue));
			}
			catch (Exception e) {
				log.error("Can't cast {} value to {}", stringValue, type);
			}
		}
		return result;
	}

	protected <T> T getJson(String paramName, Class<T> clazz) {
		final String stringValue = StringEscapeUtils.unescapeHtml4(getString(paramName));
		if (StringUtils.isNoneEmpty(stringValue)) {
			try {
				return JsonUtils.jsonToObject(stringValue, clazz);
			}
			catch (JsonProcessingException e) {
				return null;
			}
		}
		return null;
	}

	public String getParam(String paramName) {
		return params.get(paramName);
	}

	protected Map<String, String> getParams() {
		return params;
	}
}
