package host.anzo.commons.utils;

import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.*;

/**
 * @author ANZO
 * @since 12.09.2013
 */
@Slf4j
public class ClassUtils {
    public static Object singletonInstance(Class<?> clazz) {
        try {
            Method method = clazz.getDeclaredMethod("getInstance");
            return method.invoke(null);
        } catch (Exception e) {
            e.printStackTrace();
            System.exit(0);
            throw null;
        }
    }

    public static Object singletonInstanceMethod(Class<?> clazz, Method method) {
        try {
            final Object singletonInstance = singletonInstance(clazz);
            if (singletonInstance != null) {
                return method.invoke(singletonInstance);
            }
            return null;
        } catch (Exception e) {
            e.printStackTrace();
            System.exit(0);
            return null;
        }
    }

    public static @NotNull Collection<Method> getMethodsAnnotatedWith(final Class<?> type, final Class<? extends Annotation> annotation) {
        final Map<String, Method> methods = new HashMap<>();
        Class<?> klass = type;
        while (klass != Object.class) {
            final List<Method> allMethods = Arrays.stream(klass.getMethods()).filter(method -> method.isAnnotationPresent(annotation)).toList();
            for (Method method : allMethods) {
                if (!methods.containsKey(method.getName())) {
                    methods.put(method.getName(), method);
                }
                else {
                    log.error("Class [{}] contains duplicate method [{}] (original class [{}])", klass.getName(), method.getName(), type);
                }
            }
            klass = klass.getSuperclass();
        }
        return methods.values();
    }

    public static @Nullable Class<?> lookupClassQuietly(String name) {
        try {
            return ClassUtils.class.getClassLoader().loadClass(name);
        }
        catch (Exception ignored) {
        }
        return null;
    }

    public static @NotNull Map<String, Object> getFieldValues(final @NotNull Class<?> clazz) {
        final Map<String, Object> fieldValues = new HashMap<>();
        for (Field field : clazz.getDeclaredFields()) {
            try {
                fieldValues.put(field.getName(), field.get(null));
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        }
        return fieldValues;
    }

    public static void initFieldsFromMap(final Object object, @NotNull Map<String, Object> fieldValues) {
        for (Map.Entry<String, Object> entry : fieldValues.entrySet()) {
            final String fieldName = entry.getKey();
            final Object fieldValue = entry.getValue();
            final Field objectField;
            try {
                objectField = object.getClass().getField(fieldName);
                objectField.set(object, fieldValue);
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
            catch (NoSuchFieldException ignored) {
            }
        }
    }

    public static @Nullable <T> Constructor<T> getConstructor(final @NotNull Class<T> clazz, Class<?>... parameterTypes) {
        for (Constructor<?> constructor : clazz.getConstructors()) {
            if (constructor.getParameterTypes().length != parameterTypes.length) {
                continue;
            }
            boolean paramsMatch = true;
            for (int paramIndex = 0; paramIndex < parameterTypes.length; paramIndex++) {
                if (!constructor.getParameterTypes()[paramIndex].equals(parameterTypes[paramIndex])) {
                    paramsMatch = false;
                    break;
                }
            }
            if (paramsMatch) {
                return (Constructor<T>)constructor;
            }
        }
        return null;
    }

    public static boolean hasConstructor(final @NotNull Class<?> clazz, Class<?>... parameterTypes) {
        return getConstructor(clazz, parameterTypes) != null;
    }
}