package host.anzo.commons.emergency.deadlock;

import host.anzo.core.startup.StartupComponent;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;

import java.lang.management.*;
import java.util.*;

/**
 * @author ANZO
 * @since 08.07.2017
 */
@Slf4j
@StartupComponent("Diagnostic")
@SuppressWarnings("unused")
public class DeadlockDetector {
	@Getter(lazy = true)
	private final static DeadlockDetector instance = new DeadlockDetector();

	private final ThreadMXBean mbean = ManagementFactory.getThreadMXBean();
	private final List<IDeadlockListener> listeners = new ArrayList<>();

	private boolean isDeadlockFound = false;

	private DeadlockDetector() {
		new Timer("Deadlock detector", true).schedule(new TimerTask() {
			@Override
			public void run() {
				long[] ids = mbean.findMonitorDeadlockedThreads();
				if (ids != null && ids.length > 0 && !isDeadlockFound) {
					isDeadlockFound = true;

					ThreadInfo[] threadInfos = mbean.getThreadInfo(ids, true, true);
					StringBuilder info = new StringBuilder();
					info.append("DeadLock Found!\n");
					for (ThreadInfo threadInfo : threadInfos) {
						info.append(threadInfo.toString());
					}

					for (ThreadInfo ti : threadInfos) {
						LockInfo[] locks = ti.getLockedSynchronizers();
						MonitorInfo[] monitors = ti.getLockedMonitors();
						if ((locks.length == 0) && (monitors.length == 0)) {
							continue;
						}

						ThreadInfo dl = ti;
						info.append("Java-level deadlock:\n");
						info.append('\t');
						info.append(dl.getThreadName());
						info.append(" is waiting to lock ");
						info.append(dl.getLockInfo().toString());
						info.append(" which is held by ");
						info.append(dl.getLockOwnerName()).append("\n");
						while ((dl = mbean.getThreadInfo(new long[]
								{
										dl.getLockOwnerId()
								}, true, true)[0]).getThreadId() != ti.getThreadId()) {
							info.append('\t');
							info.append(dl.getThreadName());
							info.append(" is waiting to lock ");
							info.append(dl.getLockInfo().toString());
							info.append(" which is held by ");
							info.append(dl.getLockOwnerName()).append("\n");
						}
					}

					System.out.println(info);
					log.warn(info.toString());

					listeners.forEach(listener -> listener.deadlockDetected(threadInfos));

					killDeadlockedThreads(ids);
				}
			}
		}, 0, 5000);
		log.info("DeadLockDetector started.");
	}

	private void killDeadlockedThreads(long[] ids) {
		for(Map.Entry<Thread, StackTraceElement[]> entry : Thread.getAllStackTraces().entrySet()) {
			if (ArrayUtils.contains(ids, entry.getKey().getId())) {
				entry.getKey().interrupt();
			}
		}
	}

	public void registerListener(IDeadlockListener listener) {
		listeners.add(listener);
	}
}