package host.anzo.commons.utils;

import org.apache.commons.text.TextStringBuilder;
import org.jetbrains.annotations.NotNull;

import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAdjusters;
import java.util.concurrent.TimeUnit;

/**
 * @author ANZO
 * @since 9/20/2018
 */
public class DateTimeUtils {
	public static final LocalDateTime MAX_SQL_DATE = LocalDateTime.of(2099, 1, 1, 0, 0, 0);

	private static final DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

	public static long toEpochMillis(@NotNull LocalDateTime localDateTime) {
		return localDateTime.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
	}

	public static String getInternationalTime(long millis) {
		final TextStringBuilder builder = new TextStringBuilder();
		builder.append("[SYS: ").append(LocalDateTime.ofInstant(Instant.ofEpochMilli(millis), ZoneId.systemDefault()).format(dateFormatter)).append("] / ");
		builder.append("[EDT: ").append(LocalDateTime.ofInstant(Instant.ofEpochMilli(millis), ZoneId.of("GMT-04:00")).format(dateFormatter)).append("] / ");
		builder.append("[CET: ").append(LocalDateTime.ofInstant(Instant.ofEpochMilli(millis), ZoneId.of("GMT+02:00")).format(dateFormatter)).append("]");
		return builder.get();
	}

	public static @NotNull String getFormattedDateTime(@NotNull LocalDateTime localDateTime) {
		return localDateTime.format(dateFormatter);
	}

	/**
	 * @param date1 date in epoch format
	 * @param date2 date in epoch format
	 * @return {@code true} if date1 is same day as date2, {@code false} otherwise
	 */
	public static boolean isSameDay(long date1, long date2) {
		final LocalDateTime localDateTime1 = LocalDateTime.ofInstant(Instant.ofEpochMilli(date1), ZoneId.systemDefault());
		final LocalDateTime localDateTime2 = LocalDateTime.ofInstant(Instant.ofEpochMilli(date2), ZoneId.systemDefault());
		return localDateTime1.getDayOfYear() == localDateTime2.getDayOfYear();
	}

	public static int parseTimeSpan(@NotNull String timeSpan, TimeUnit unit) {
		final String[] timeSpanData = timeSpan.split(":");
		if (timeSpanData.length == 3) {
			int resultSeconds = Integer.parseInt(timeSpanData[0]) * 3600;
			resultSeconds += Integer.parseInt(timeSpanData[1]) * 60;
			resultSeconds += Integer.parseInt(timeSpanData[2]);
			return (int)unit.convert(resultSeconds, TimeUnit.SECONDS);
		}
		return -1;
	}

	public static String toTimeSpan(long seconds) {
		return String.format("%02d:%02d:%02d", seconds / 3600, (seconds % 3600) / 60, seconds % 60);
	}

	/***
	 * @param day day of week when SUNDAY=0 and MONDAY=6
	 * @return DayOfWeek enum value
	 */
	public static DayOfWeek parseDay(int day) {
		if (day == 0) {
			return DayOfWeek.SUNDAY;
		}
		return DayOfWeek.values()[day - 1];
	}

	public static LocalDateTime getLocalDateTime() {
		return LocalDateTime.ofInstant(Instant.ofEpochMilli(System.currentTimeMillis()), ZoneId.systemDefault());
	}

	public static LocalDateTime getLocalDateTime(long epochTime) {
		return LocalDateTime.ofInstant(Instant.ofEpochMilli(epochTime), ZoneId.systemDefault());
	}

	public static ZonedDateTime getZonedDateTime(long epochTime) {
		return ZonedDateTime.ofInstant(Instant.ofEpochMilli(epochTime), ZoneId.systemDefault());
	}

	public static LocalDate getLocalDate(long epochTime) {
		return LocalDateTime.ofInstant(Instant.ofEpochMilli(epochTime), ZoneId.systemDefault()).toLocalDate();
	}

	/***
	 * @param timeInMillis epoch time stamp
	 * @param timeUnit time unit
	 * @return time passed from current day start in specified time unit
	 */
	public static long getCurrentDayPassedTime(long timeInMillis, @NotNull TimeUnit timeUnit) {
		return timeUnit.convert(timeInMillis - getLocalDate(timeInMillis).atStartOfDay().atZone(ZoneId.systemDefault()).toInstant().toEpochMilli(), TimeUnit.MILLISECONDS);
	}

	/***
	 * @param timeInMillis epoch time stamp
	 * @param hour specified hour
	 * @return time at specified hour at current day
	 */
	public static long getCurrentDayHourTime(long timeInMillis, int hour) {
		return getLocalDateTime(timeInMillis).withHour(0).withMinute(0).withSecond(0).plusHours(hour).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
	}

	/***
	 * @param timeInMillis epoch time stamp
	 * @return current month start epoch time
	 */
	public static long getCurrentMonthStartTime(long timeInMillis) {
		final LocalDate localDate = getLocalDate(timeInMillis);
		final LocalDate currentMonthStartDay = LocalDate.of(localDate.getYear(), localDate.getMonth(), 1);
		return currentMonthStartDay.atStartOfDay(ZoneId.systemDefault()).toInstant().toEpochMilli();
	}

	/***
	 * @param timeInMillis epoch time stamp
	 * @param hour specified hour
	 * @return time at specified next hour
	 */
	public static long getNextHourTime(long timeInMillis, int hour) {
		long nextHour = getLocalDateTime(timeInMillis).withHour(0).withMinute(0).withSecond(0).plusHours(hour).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
		return nextHour < getLocalDateTime(timeInMillis).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli() ? getNextDayHourTime(timeInMillis, hour) : nextHour;
	}

	public static long getNextHourTime(long timeInMillis) {
		final LocalDateTime localDateTime = getLocalDateTime(timeInMillis);
		return getNextHourTime(timeInMillis, localDateTime.getHour() + 1);
	}

	/***
	 * @param timeInMillis epoch time stamp
	 * @param dayOfWeek day of week
	 * @return next specified DayOfWeek start epoch time
	 */
	public static long getNextDayStartTime(long timeInMillis, DayOfWeek dayOfWeek) {
		return getLocalDateTime(timeInMillis).with(TemporalAdjusters.next(dayOfWeek)).withHour(0).withMinute(0).withSecond(0).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
	}

	/***
	 * @param timeInMillis epoch time stamp
	 * @return next day start epoch time
	 */
	public static long getNextDayStartTime(long timeInMillis) {
		return getNextDayHourTime(timeInMillis, 0);
	}

	/***
	 * @param timeInMillis epoch time stamp
	 * @param hour specified hour
	 * @return time at specified hour at next day
	 */
	public static long getNextDayHourTime(long timeInMillis, int hour) {
		return getLocalDateTime(timeInMillis).withHour(0).withMinute(0).withSecond(0).plusDays(1).plusHours(hour).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
	}

	/***
	 * @param timeInMillis epoch time stamp
	 * @return next month start epoch time
	 */
	public static long getNextMonthStartTime(long timeInMillis) {
		final LocalDate localDate = getLocalDate(timeInMillis);
		final LocalDate nextMonthDay = LocalDate.of(localDate.getYear(), localDate.getMonth(), 1).plus(1, ChronoUnit.MONTHS);
		return nextMonthDay.atStartOfDay(ZoneId.systemDefault()).toInstant().toEpochMilli();
	}

	/***
	 * @param timeInMillis epoch time stamp
	 * @param dayOfWeek day of week
	 * @return time in millis to next specified DayOfWeek start
	 */
	public static long getTimeToNextDayStart(long timeInMillis, DayOfWeek dayOfWeek) {
		return Math.max(0, getNextDayStartTime(timeInMillis, dayOfWeek) - getLocalDateTime(timeInMillis).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli());
	}

	/***
	 * @param timeInMillis epoch time stamp
	 * @return time to next day start
	 */
	public static long getTimeToNextDayStart(long timeInMillis) {
		return Math.max(0, getNextDayStartTime(timeInMillis) - getLocalDateTime(timeInMillis).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli());
	}

	/**
	 * @param dateTime date time to calculate time to
	 * @param timeUnit time unit
	 * @return time to specified date time in specified time unit
	 */
	public static long getTimeTo(@NotNull LocalDateTime dateTime, @NotNull TimeUnit timeUnit) {
		return timeUnit.convert(Math.max(0, dateTime.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli() - LocalDateTime.now().atZone(ZoneId.systemDefault()).toInstant().toEpochMilli()), TimeUnit.MILLISECONDS);
	}
}