package host.anzo.commons.versioning;

import host.anzo.commons.utils.ConsoleUtils;
import host.anzo.commons.utils.DateTimeUtils;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;

import java.io.File;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.jar.Attributes;
import java.util.jar.JarFile;

/**
 * @author ANZO
 * @since 06.07.2016
 */
@Slf4j
public class Version {
	@Getter(lazy = true)
	private static final Version instance = new Version();

	private @Getter String versionRevision;
	private @Getter String buildDate = "";
	private @Getter String buildJdk = "";

	public void init(Class<?> c) {
		File jarName = null;
		try {
			jarName = Locator.getClassSource(c);

			// Only if we starting from Jar
			if (StringUtils.endsWith(jarName.getName(), ".jar")) {
				try (JarFile jarFile = new JarFile(jarName)) {
					Attributes attrs = jarFile.getManifest().getMainAttributes();
					setBuildJdk(attrs);
					setBuildDate(attrs);
					setVersionRevision(attrs);
				}
			}
			else {
				versionRevision = "Dev";
				buildDate = DateTimeUtils.getFormattedDateTime(LocalDateTime.now());
				buildJdk = System.getProperty("java.version");
			}
		} catch (IOException e) {
			log.error("Unable to get soft information. File name '{}' isn't a valid jar", jarName.getAbsolutePath(), e);
		}
		finally {
			info();
		}
	}

	/**
	 * @param attrs manifest attributes
	 */
	private void setVersionRevision(@NotNull Attributes attrs) {
		String versionRevision = attrs.getValue("Implementation-Version");
		if (versionRevision != null && !versionRevision.isEmpty()) {
			this.versionRevision = versionRevision;
		} else {
			this.versionRevision = "N/A";
		}
	}

	/**
	 * @param attrs manifest attributes
	 */
	private void setBuildJdk(@NotNull Attributes attrs) {
		String buildJdk = attrs.getValue("Built-JDK");
		if (buildJdk != null) {
			this.buildJdk = buildJdk;
		} else {
			buildJdk = attrs.getValue("Source-Compatibility");
			if (buildJdk != null) {
				this.buildJdk = buildJdk;
			} else {
				this.buildJdk = "-1";
			}
		}
	}

	/**
	 * @param attrs manifest attributes
	 */
	private void setBuildDate(@NotNull Attributes attrs) {
		String buildDate = attrs.getValue("Built-Date");
		if (buildDate != null) {
			this.buildDate = buildDate;
		} else {
			this.buildDate = "-1";
		}
	}

	private void info() {
		ConsoleUtils.printSection("Build Information");
		log.info("Revision: ................ {}", getVersionRevision());
		log.info("Build date: .............. {}", getBuildDate());
	}

	@Override
	public String toString() {
		return "Version " + versionRevision + " from " + buildDate;
	}
}