package hudson.plugins.sctmexecutor;

import hudson.model.Descriptor;
import hudson.model.Hudson;
import hudson.plugins.sctmexecutor.validators.EmptySingleFieldValidator;
import hudson.plugins.sctmexecutor.validators.NumberCSVSingleFieldValidator;
import hudson.plugins.sctmexecutor.validators.TestConnectionValidator;
import hudson.tasks.Builder;
import hudson.util.FormFieldValidator;

import java.io.BufferedReader;
import java.io.IOException;
import java.net.URL;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.servlet.ServletException;

import net.sf.json.JSONObject;

import org.kohsuke.stapler.QueryParameter;
import org.kohsuke.stapler.StaplerRequest;
import org.kohsuke.stapler.StaplerResponse;

/**
 * 
 * @author Thomas Fuerer
 * 
 */
public final class SCTMExecutorDescriptor extends Descriptor<Builder> {
  private String serviceURL;
  private String user;
  private String password;
  private ExecutorService threadPool;

  SCTMExecutorDescriptor() {
    super(SCTMExecutor.class);
    load();
  }

  @Override
  public String getDisplayName() {
    return Messages.getString("SCTMExecutorDescriptor.plugin.title"); //$NON-NLS-1$
  }

  @Override
  public Builder newInstance(StaplerRequest req, JSONObject formData) throws FormException {
    String execDefIds = formData.getString("execDefIds"); //$NON-NLS-1$
    String str = formData.getString("projectId"); //$NON-NLS-1$
    int projectId = Integer.parseInt(str);
    return new SCTMExecutor(projectId, execDefIds);
  }

  @Override
  public boolean configure(StaplerRequest req, JSONObject json) throws hudson.model.Descriptor.FormException {
    serviceURL = json.getString("serviceURL"); //$NON-NLS-1$
    user = json.getString("user"); //$NON-NLS-1$
    password = PwdCrypt.encode(json.getString("password"), Hudson.getInstance().getSecretKey()); //$NON-NLS-1$

    save();
    return super.configure(req, json);
  }

  public void setServiceURL(String serviceURL) {
    this.serviceURL = serviceURL;
  }

  public String getServiceURL() {
    return serviceURL;
  }

  public String getUser() {
    return user;
  }

  public String getPassword() {
    if (password != null && !password.equals(""))
      return PwdCrypt.decode(password, Hudson.getInstance().getSecretKey());
    else
      return "";
  }

  public void setUser(String user) {
    this.user = user;
  }

  public void setPassword(String password) {
    this.password = password;
  }

  public void doCheckServiceURL(StaplerRequest req, StaplerResponse rsp, 
      @QueryParameter("value") final String value)
      throws IOException, ServletException {
    new FormFieldValidator.URLCheck(req, rsp) {
      @Override
      protected void check() throws IOException, ServletException {
        if (value == null ||
            (value != null && !value.matches("http(s)?://(((\\d{1,3}.){3}\\d{1,3})?|([\\p{Alnum}-_.])*)(:\\d{0,5})?(/([\\p{Alnum}-_.])*)?/services"))) { //$NON-NLS-1$
          error(Messages.getString("SCTMUrlValidator.msg.noValidURL")); //$NON-NLS-1$
          return;
        }
        try {
          URL url = new URL(value);
          BufferedReader reader = open(url);
          if (findText(reader, "tmexecution"))
            ok();
          else
            warning(Messages.getString("SCTMURL.Validator.msg.noServiceFound"));
        } catch (IOException e) {
          handleIOException(value, e);
        } catch (IllegalArgumentException e) {
          error(Messages.getString("SCTMUrlValidator.msg.noValidURL")); //$NON-NLS-1$
          
        }
      }
    }.process();
  }

  public void doCheckUser(StaplerRequest req, StaplerResponse rsp, 
      @QueryParameter("value") final String value)
      throws IOException, ServletException {
    new EmptySingleFieldValidator(value).process();
  }

  public void doCheckPassword(StaplerRequest req, StaplerResponse rsp, 
      @QueryParameter("value") final String value)
      throws IOException, ServletException {
    new EmptySingleFieldValidator(value).process();
  }

  public void doCheckExecDefIds(StaplerRequest req, StaplerResponse rsp, 
      @QueryParameter("value") final String value)
      throws IOException, ServletException {
    new NumberCSVSingleFieldValidator(value).process();
  }

  public void doCheckProjectId(StaplerRequest req, StaplerResponse rsp, 
      @QueryParameter("value") final String value)
      throws IOException, ServletException {
    new FormFieldValidator.NonNegativeInteger().process();
  }

  public void doTestConnection(StaplerRequest req, StaplerResponse rsp, 
      @QueryParameter("serviceURL") final String serviceURL,
      @QueryParameter("user") final String user,
      @QueryParameter("password") final String password) throws IOException, ServletException {
    new TestConnectionValidator(serviceURL, user, password).process();
  }

  public ExecutorService getExecutorPool() {
    if (threadPool == null || threadPool.isShutdown() || threadPool.isTerminated())
      threadPool = Executors.newFixedThreadPool(8);
    return threadPool;
  }

  public void shutdown() {
    if (threadPool != null)
      threadPool.shutdown();
  }
}