// Copyright (c) 2020-2021 DragonKnightOfBreeze Windea
// Breeze is blowing...

package icu.windea.breezeframework.core.component

/**
 * 字符串模式。
 *
 * 字符串模式用于表示字符串符合某种格式或规则。
 */
interface StringPattern : Component {
	/**
	 * 判断指定的字符串是否匹配。
	 */
	fun matches(value: String): Boolean

	companion object Registry : AbstractComponentRegistry<StringPattern>() {
		override fun registerDefault() {
			register(BooleanPattern)
			register(WildcardBooleanPattern)
			register(IntPattern)
			register(NumberPattern)
			register(AlphaPattern)
			register(NumericPattern)
			register(AlphanumericPattern)
		}
	}

	//region String Patterns
	object BooleanPattern : StringPattern {
		private val values = arrayOf("true", "false")

		override fun matches(value: String): Boolean {
			return value in values
		}
	}

	object WildcardBooleanPattern : StringPattern {
		private val values = arrayOf("true", "false", "yes", "no", "on", "off")

		override fun matches(value: String): Boolean {
			return value in values
		}
	}

	object IntPattern : StringPattern {
		private val signs = charArrayOf('+', '-')

		override fun matches(value: String): Boolean {
			val sign = value.firstOrNull()
			val noSignValue = if(sign == null) value else if(sign in signs) value.drop(1) else return false
			return noSignValue.any { it.isDigit() }
		}
	}

	object NumberPattern : StringPattern {
		private val signs = charArrayOf('+', '-')

		override fun matches(value: String): Boolean {
			val sign = value.firstOrNull()
			val noSignValue = if(sign == null) value else if(sign in signs) value.drop(1) else return false
			var hasDot = false
			return noSignValue.any {
				if(it == '.') {
					if(!hasDot) hasDot = true else return false
				}
				it.isDigit()
			}
		}
	}

	object AlphaPattern : StringPattern {
		override fun matches(value: String): Boolean {
			return value.isNotEmpty() && value.all { it.isLetter() }
		}
	}

	object NumericPattern : StringPattern {
		override fun matches(value: String): Boolean {
			return value.isNotEmpty() && value.all { it.isDigit() }
		}
	}

	object AlphanumericPattern : StringPattern {
		override fun matches(value: String): Boolean {
			return value.isNotEmpty() && value.all { it.isLetterOrDigit() }
		}
	}

	object BlankPattern:StringPattern{
		override fun matches(value: String): Boolean {
			return value.isBlank()
		}
	}
	//endregion
}

