package icu.wuhufly
import org.apache.spark.sql.{DataFrame, SparkSession}

class SparkHandler(create: Create, jdbcOutput: JdbcOutput, jdbcInput: JdbcInput, hdfsio: HDFSIO, merge: Merge, increment: Increment)
  extends Create with JdbcOutput with JdbcInput with HDFSIO with Merge with Increment
{
  override def getSpark(setMaster: Boolean = false): SparkSession = create.getSpark(setMaster)

  override def writeIntoMysql(tbName: String, df: DataFrame, dbName: String = "shtd_industry", overwrite: Boolean = true): Unit = jdbcOutput.writeIntoMysql(tbName, df, dbName, overwrite)

  override def writeIntoCK(tbName: String, df: DataFrame, orderBy: String, dbName: String = "shtd_industry", overwrite: Boolean = true): Unit = jdbcOutput.writeIntoCK(tbName, df, orderBy, dbName, overwrite)

  override def readFromMysql(tbName: String, spark: SparkSession, dbName: String = "shtd_industry"): DataFrame = jdbcInput.readFromMysql(tbName, spark, dbName)

  override def readFromCK(tbName: String, spark: SparkSession, dbName: String = "shtd_industry"): DataFrame = jdbcInput.readFromCK(tbName, spark, dbName)

  override def writeIntoHDFS(tbName: String, df: DataFrame, recordKey: String, preCombineKey: String,
                             partitionKey: String = "etldate", dbName: String = "hudi_gy_ods", overwrite: Boolean = true, sync: Boolean = true): Unit = hdfsio.writeIntoHDFS(tbName, df, recordKey, preCombineKey, partitionKey, dbName, overwrite, sync)

  override def readFromHDFS(tbName: String, spark: SparkSession, dbName: String = "hudi_gy_ods"): DataFrame = hdfsio.readFromHDFS(tbName, spark, dbName)

  override def mergeIntoHDFS(tbName: String, newDf: DataFrame, recordKey: String, preCombineKey: String,
                             partitionKey: String = "etldate", dbName: String = "hudi_gy_dwd"): Unit = merge.mergeIntoHDFS(tbName, newDf, recordKey, preCombineKey, partitionKey, dbName)

  override def findIncremental(tbName: String, compareKey: Array[String],
                               df: DataFrame, default: String, dbName: String = "hudi_gy_dwd"): DataFrame = increment.findIncremental(tbName, compareKey, df, default, dbName)
}

object SparkHandler {
  def of(): SparkHandler = {
    val create: Create = Class.forName("icu.wuhufly.impl.CreateImpl").getConstructor().newInstance().asInstanceOf[Create]
    val input: JdbcInput = Class.forName("icu.wuhufly.impl.JdbcInputImpl").getConstructor().newInstance().asInstanceOf[JdbcInput]
    val output: JdbcOutput = Class.forName("icu.wuhufly.impl.JdbcOutputImpl").getConstructor().newInstance().asInstanceOf[JdbcOutput]
    val io: HDFSIO = Class.forName("icu.wuhufly.impl.HDFSIOImpl").getConstructor().newInstance().asInstanceOf[HDFSIO]
    val merge: Merge = Class.forName("icu.wuhufly.impl.MergeImpl").getConstructor(classOf[HDFSIO]).newInstance(io).asInstanceOf[Merge]
    val increment: Increment = Class.forName("icu.wuhufly.impl.IncrementImpl").getConstructor().newInstance().asInstanceOf[Increment]
    new SparkHandler(create = create, jdbcOutput = output, jdbcInput = input, hdfsio = io, merge = merge, increment = increment)
  }
}