package `in`.androidplay.pollingengine.polling

/**
 * Predefined BackoffPolicy configurations for common scenarios.
 *
 * Use these to avoid tuning by hand for typical needs. You can always pass a custom [BackoffPolicy]
 * when necessary.
 */
public object BackoffPolicies {
    /**
     * Quick polling tuned for short-lived availability (e.g., compliance status) with ~20s cap.
     */
    public val quick20s: BackoffPolicy = BackoffPolicy(
        initialDelayMs = 500,
        maxDelayMs = 5_000,
        multiplier = 1.8,
        jitterRatio = 0.2,
        maxAttempts = 20,
        overallTimeoutMs = 20_000,
        perAttemptTimeoutMs = 10_000,
    )

    /**
     * Balanced exponential backoff suitable for many network APIs.
     * Roughly doubles up to 30s and stops around 2 minutes.
     */
    public val exponentialDefault: BackoffPolicy = BackoffPolicy(
        initialDelayMs = 1_000,
        maxDelayMs = 30_000,
        multiplier = 2.0,
        jitterRatio = 0.2,
        maxAttempts = 8,
        overallTimeoutMs = 120_000,
        perAttemptTimeoutMs = null,
    )

    /**
     * Gentle linear(ish) backoff for UIs that need frequent updates without spamming servers.
     * Lower multiplier approximates linear growth over many attempts.
     */
    public val linearGentle: BackoffPolicy = BackoffPolicy(
        initialDelayMs = 1_000,
        maxDelayMs = 10_000,
        multiplier = 1.2,
        jitterRatio = 0.1,
        maxAttempts = 30,
        overallTimeoutMs = 180_000,
        perAttemptTimeoutMs = null,
    )

    /**
     * Very aggressive and short-lived for debugging or tests. Not recommended for production.
     */
    public val rapidDebug: BackoffPolicy = BackoffPolicy(
        initialDelayMs = 100,
        maxDelayMs = 1_000,
        multiplier = 1.3,
        jitterRatio = 0.0,
        maxAttempts = 10,
        overallTimeoutMs = 5_000,
        perAttemptTimeoutMs = 2_000,
    )
}
