package `in`.androidplay.pollingengine.polling

import `in`.androidplay.pollingengine.polling.builder.PollingConfigBuilder
import kotlinx.coroutines.flow.Flow

/**
 * Public abstraction layer for consumers. Exposes a small, stable API surface.
 * Internals are delegated to the PollingEngine implementation.
 *
 * Plain-English entrypoints:
 * - startPolling { ... } — start and observe via Flow
 * - doPolling { ... } — alias for startPolling, reads more naturally in code
 * - run(config) — run synchronously (suspending) and return a terminal outcome
 */
public interface PollingApi {
    /** Number of active polling sessions. */
    public fun activePollsCount(): Int

    /** IDs of active polling sessions. */
    public suspend fun listActiveIds(): List<String>

    /** Cancel a session by ID. */
    public suspend fun cancel(id: String)

    /** Cancel a session using the session handle. */
    public suspend fun cancel(session: PollingSession)

    /** Cancel all active sessions. */
    public suspend fun cancelAll()

    /** Shutdown the SDK's internal engine. After shutdown, new sessions cannot be started. */
    public suspend fun shutdown()

    /** Pause a session by ID. */
    public suspend fun pause(id: String)

    /** Resume a session by ID. */
    public suspend fun resume(id: String)

    /** Update backoff/options for a running session by ID. */
    public suspend fun updateBackoff(id: String, newPolicy: BackoffPolicy)

    /** Start a new polling session using the fluent builder DSL. */
    public fun <T> startPolling(
        builder: PollingConfigBuilder<T>.() -> Unit
    ): Flow<PollingOutcome<T>>

    /**
     * Alias for startPolling that reads more naturally in codebases.
     * Example: Polling.doPolling { fetch = { ... } }
     */
    public fun <T> doPolling(
        builder: PollingConfigBuilder<T>.() -> Unit
    ): Flow<PollingOutcome<T>> = startPolling(builder)

    /** One-shot polling that runs to completion synchronously (suspending). */
    public suspend fun <T> run(config: PollingConfig<T>): PollingOutcome<T>

    /** Compose multiple polling configs sequentially. */
    public suspend fun <T> compose(vararg configs: PollingConfig<T>): PollingOutcome<T>
}
