package in.clouthink.daas.bm.support.mongodb.repository.custom.impl;

import in.clouthink.daas.bm.domain.request.BusinessMessageQueryRequest;
import in.clouthink.daas.bm.support.mongodb.model.BusinessMessageEntity;
import in.clouthink.daas.bm.support.mongodb.repository.custom.BusinessMessageRepositoryCustom;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import java.util.Date;
import java.util.List;

import static org.springframework.data.mongodb.core.query.Criteria.where;
import static org.springframework.data.mongodb.core.query.Query.query;

@Repository
public class BusinessMessageRepositoryImpl implements BusinessMessageRepositoryCustom {

	@Autowired
	protected MongoTemplate mongoTemplate;

	@Override
	public long count(BusinessMessageQueryRequest queryRequest) {
		Query query = buildQuery(queryRequest);
		return mongoTemplate.count(query, BusinessMessageEntity.class);
	}

	@Override
	public Page<BusinessMessageEntity> queryPage(BusinessMessageQueryRequest queryRequest) {
		Query query = buildQuery(queryRequest);

		long count = mongoTemplate.count(query, BusinessMessageEntity.class);

		int start = queryRequest.getStart();
		int limit = queryRequest.getLimit();

		PageRequest pageable = new PageRequest(start, limit, new Sort(Sort.Direction.DESC, "sendAt"));
		query.with(pageable);

		List<BusinessMessageEntity> list = mongoTemplate.find(query, BusinessMessageEntity.class);

		return new PageImpl<>(list, pageable, count);
	}

	@Override
	public void markAsRead(String id) {
		mongoTemplate.updateFirst(query(where("id").is(id)),
								  new Update().set("read", true).set("readAt", new Date()),
								  BusinessMessageEntity.class);
	}

	@Override
	public void markAsProcessed(String id) {
		mongoTemplate.updateFirst(query(where("id").is(id)),
								  new Update().set("processed", true).set("processAt", new Date()),
								  BusinessMessageEntity.class);
	}

	@Override
	public void markAsDeleted(String id) {
		mongoTemplate.updateFirst(query(where("id").is(id)),
								  new Update().set("deleted", true).set("deleteAt", new Date()),
								  BusinessMessageEntity.class);

	}

	private Query buildQuery(BusinessMessageQueryRequest queryRequest) {
		Query query = new Query();


		if (!StringUtils.isEmpty(queryRequest.getCategory())) {
			query.addCriteria(where("category").regex(queryRequest.getCategory()));
		}

		if (queryRequest.getReceiverId() != null) {
			query.addCriteria(where("receiverId").is(queryRequest.getReceiverId()));
		}

		if (!StringUtils.isEmpty(queryRequest.getReceiverName())) {
			query.addCriteria(where("receiverName").regex(queryRequest.getReceiverName()));
		}

		if (queryRequest.getRead() != null) {
			query.addCriteria(where("read").ne(!queryRequest.getRead()));
		}

		if (queryRequest.getDeleted() != null) {
			query.addCriteria(where("deleted").ne(!queryRequest.getDeleted()));
		}

		if (queryRequest.getBeginDate() != null && queryRequest.getEndDate() != null) {
			query.addCriteria(new Criteria().andOperator(Criteria.where("sendAt").gte(queryRequest.getBeginDate()),
														 Criteria.where("sendAt").lte(queryRequest.getEndDate())));
		}
		else if (queryRequest.getBeginDate() != null) {
			query.addCriteria(Criteria.where("sendAt").gte(queryRequest.getBeginDate()));
		}
		else if (queryRequest.getEndDate() != null) {
			query.addCriteria(Criteria.where("sendAt").lte(queryRequest.getEndDate()));
		}

		return query;
	}

}
