package in.clouthink.daas.bm.support.mongodb.spiImpl;

import in.clouthink.daas.bm.core.MessageReceiver;
import in.clouthink.daas.bm.domain.model.BusinessMessage;
import in.clouthink.daas.bm.domain.model.FailureMessage;
import in.clouthink.daas.bm.domain.request.BusinessMessageQueryRequest;
import in.clouthink.daas.bm.domain.request.FailureMessageQueryRequest;
import in.clouthink.daas.bm.domain.service.BusinessMessageService;
import in.clouthink.daas.bm.exception.BusinessMessageNotFoundException;
import in.clouthink.daas.bm.exception.BusinessMessagePermissionException;
import in.clouthink.daas.bm.support.mongodb.model.BusinessMessageEntity;
import in.clouthink.daas.bm.support.mongodb.model.FailureMessageEntity;
import in.clouthink.daas.bm.support.mongodb.repository.BusinessMessageRepository;
import in.clouthink.daas.bm.support.mongodb.repository.FailureMessageRepository;
import in.clouthink.daas.edm.Listenable;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;

import java.util.stream.Collectors;


public class BusinessMessageServiceImpl implements BusinessMessageService {

	@Autowired
	private BusinessMessageRepository businessMessageRepository;

	@Autowired
	private FailureMessageRepository failureMessageRepository;

	@Listenable
	@Override
	public BusinessMessage saveBusinessMessage(BusinessMessage message) {
		if (message instanceof BusinessMessageEntity) {
			return businessMessageRepository.save((BusinessMessageEntity) message);
		}
		BusinessMessageEntity entity = new BusinessMessageEntity();
		BeanUtils.copyProperties(message, entity);
		return businessMessageRepository.save(entity);
	}

	@Override
	public BusinessMessage findBusinessMessageById(String messageId) {
		return businessMessageRepository.findById(messageId);
	}

	@Override
	public Page<BusinessMessage> listBusinessMessages(BusinessMessageQueryRequest queryRequest) {
		Page<BusinessMessageEntity> resultPage = businessMessageRepository.queryPage(queryRequest);
		return new PageImpl<>(resultPage.getContent()
										.stream()
										.map(item -> (BusinessMessage<String>) item)
										.collect(Collectors.toList()),
							  new PageRequest(queryRequest.getStart(), queryRequest.getLimit()),
							  resultPage.getTotalElements());
	}

	@Override
	public void markAsRead(String messageId, MessageReceiver receiver) {
		if (receiver == null) {
			throw new BusinessMessagePermissionException("Illegal access.");
		}

		BusinessMessageEntity businessMessageEntity = businessMessageRepository.findById(messageId);
		if (businessMessageEntity == null) {
			throw new BusinessMessageNotFoundException();
		}

		if (!receiver.getId().equals(businessMessageEntity.getReceiverId())) {
			throw new BusinessMessagePermissionException("Illegal access.");
		}

		businessMessageRepository.markAsRead(messageId);
	}

	@Override
	public void markAsProcessed(String messageId, MessageReceiver receiver) {
		if (receiver == null) {
			throw new BusinessMessagePermissionException("Illegal access.");
		}

		BusinessMessageEntity businessMessageEntity = businessMessageRepository.findById(messageId);
		if (businessMessageEntity == null) {
			throw new BusinessMessageNotFoundException();
		}

		if (!receiver.getId().equals(businessMessageEntity.getReceiverId())) {
			throw new BusinessMessagePermissionException("Illegal access.");
		}
		businessMessageRepository.markAsProcessed(messageId);
	}

	@Override
	public void deleteBusinessMessage(String messageId, MessageReceiver receiver) {
		if (receiver == null) {
			throw new BusinessMessagePermissionException("Illegal access.");
		}

		BusinessMessageEntity businessMessageEntity = businessMessageRepository.findById(messageId);
		if (businessMessageEntity == null) {
			throw new BusinessMessageNotFoundException();
		}

		if (!receiver.getId().equals(businessMessageEntity.getReceiverId())) {
			throw new BusinessMessagePermissionException("Illegal access.");
		}

		businessMessageRepository.markAsDeleted(messageId);
	}

	@Override
	public FailureMessage findFailureMessageById(String messageId) {
		return failureMessageRepository.findById(messageId);
	}

	@Override
	public Page<FailureMessage> listFailureMessages(FailureMessageQueryRequest queryRequest) {
		Page<FailureMessageEntity> resultPage = failureMessageRepository.queryPage(queryRequest);
		return new PageImpl<>(resultPage.getContent()
										.stream()
										.map(item -> (FailureMessage<String>) item)
										.collect(Collectors.toList()),
							  new PageRequest(queryRequest.getStart(), queryRequest.getLimit()),
							  resultPage.getTotalElements());
	}

	@Override
	public FailureMessage deleteFailureMessage(String messageId) {
		FailureMessageEntity result = failureMessageRepository.findById(messageId);
		if (result == null) {
			return null;
		}
		failureMessageRepository.delete(result);
		return result;
	}

}
