package in.clouthink.daas.sbpm.core.service;

import in.clouthink.daas.sbpm.core.exception.ActivityInstanceNotFoundException;
import in.clouthink.daas.sbpm.core.model.*;
import in.clouthink.daas.sbpm.core.spi.WorkflowRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * The abstract workflow service.
 *
 * @author dz
 */
public abstract class AbstractWorkflowService implements WorkflowService {

	@Autowired
	private WorkflowRepository workflowRepository;

	public WorkflowRepository getWorkflowRepository() {
		return workflowRepository;
	}

	public void setWorkflowRepository(WorkflowRepository workflowRepository) {
		this.workflowRepository = workflowRepository;
	}

	@Override
	public ProcessDefinition findProcessDefinitionById(String id) {
		return workflowRepository.findProcessDefinitionById(id);
	}

	@Override
	public ActivityDefinition findActivityDefinitionById(String id) {
		return workflowRepository.findActivityDefinitionById(id);
	}

	@Override
	public ActionReference getStartAction(String processDefinitionId) {
		ActivityDefinition startActivityDefinition = getStartActivityDefinition(processDefinitionId);
		return getFirstAction(startActivityDefinition);
	}

	@Override
	public ProcessInstance findProcessInstanceById(String processInstanceId) {
		return workflowRepository.findProcessInstanceById(processInstanceId);
	}

	@Override
	public ActivityInstance findActivityInstanceById(String id) {
		return workflowRepository.findActivityInstanceById(id);
	}

	@Override
	public List<ProcessHistory> listProcessHistories(String processInstanceId) {
		return workflowRepository.listProcessHistories(processInstanceId);
	}

	@Override
	public List<ActionReference> getNextActions(String activityInstanceId) {
		ActivityInstance activityInstance = workflowRepository.findActivityInstanceById(activityInstanceId);
		if (activityInstance == null) {
			throw new ActivityInstanceNotFoundException(activityInstanceId);
		}
		ActivityDefinition activityDefinition = activityInstance.getActivityDefinition();
		return getNextActions(activityDefinition);
	}

	@Override
	public Page<ActivityInstance> listActivities(ActivityQueryRequest queryRequest) {
		return workflowRepository.listActivityInstances(queryRequest);
	}

	protected ActivityDefinition getStartActivityDefinition(String processDefinitionId) {
		List<ActivityDefinition> activityDefinitionList = workflowRepository.listActivityDefinitions(processDefinitionId);
		for (ActivityDefinition activityDefinition : activityDefinitionList) {
			if (activityDefinition.getType() == ActivityType.Start) {
				return activityDefinition;
			}
		}
		return null;
	}

	protected ActionReference getFirstAction(ActivityDefinition startActivityDefinition) {
		if (startActivityDefinition != null) {
			ActivityRouter activityRouter = startActivityDefinition.getRouters().get(0);
			if (activityRouter == null) {
				return null;
			}
			DefaultActionReference result = new DefaultActionReference();
			result.setCode(activityRouter.getCode());
			result.setName(activityRouter.getName());
			return result;
		}

		return null;
	}

	protected List<ActionReference> getNextActions(ActivityDefinition activityDefinition) {
		if (activityDefinition != null) {
			return activityDefinition.getRouters().stream().map(item -> {
				DefaultActionReference result = new DefaultActionReference();
				result.setCode(item.getCode());
				result.setName(item.getName());
				return result;
			}).collect(Collectors.toList());
		}

		return Collections.emptyList();
	}

}
