package in.clouthink.daas.sbpm.core.spi;

import in.clouthink.daas.sbpm.core.model.*;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import java.util.List;

/**
 * The workflow persistence layer service.
 *
 * @author dz
 */
public interface WorkflowRepository {

	/**
	 * find process definition by id
	 *
	 * @param processDefinitionId
	 * @return
	 */
	ProcessDefinition findProcessDefinitionById(String processDefinitionId);

	/**
	 * list all process definitions
	 *
	 * @return
	 */
	List<ProcessDefinition> listProcessDefinitions();

	/**
	 * save the process definition
	 *
	 * @param request
	 * @return
	 */
	ProcessDefinition saveProcessDefinition(ProcessDefinition request);

	/**
	 * delete the process definition
	 *
	 * @param processDefinitionId
	 * @return
	 */
	ProcessDefinition deleteProcessDefinition(String processDefinitionId);

	/**
	 * find activity definition by id
	 *
	 * @param activityDefinitionId
	 * @return
	 */
	ActivityDefinition findActivityDefinitionById(String activityDefinitionId);

	/**
	 * list all activity definitions by process definition
	 *
	 * @param processDefinitionId
	 * @return
	 */
	List<ActivityDefinition> listActivityDefinitions(String processDefinitionId);

	/**
	 * list all activity definitions by process definition
	 *
	 * @param processDefinition
	 * @return
	 */
	List<ActivityDefinition> listActivityDefinitions(ProcessDefinition processDefinition);

	/**
	 * save the activity definition
	 *
	 * @param request
	 * @return
	 */
	ActivityDefinition saveActivityDefinition(ActivityDefinition request);

	/**
	 * delete the activity definition
	 *
	 * @param activityDefinitionId
	 * @return
	 */
	ActivityDefinition deleteActivityDefinition(String activityDefinitionId);

	/**
	 * find process instance by id
	 *
	 * @param instanceId
	 * @return
	 */
	ProcessInstance findProcessInstanceById(String instanceId);

	/**
	 * update the status of process instance
	 *
	 * @param processInstanceId
	 * @param processStatus
	 * @return
	 */
	ProcessInstance updateProcessInstance(String processInstanceId, ProcessStatus processStatus);

	/**
	 * list the process instance by process definition
	 *
	 * @param processDefinitionId
	 * @param pageable
	 * @return
	 */
	Page<ProcessInstance> listProcessInstances(String processDefinitionId, Pageable pageable);

	/**
	 * save process instance
	 *
	 * @param request
	 * @return
	 */
	ProcessInstance saveProcessInstance(ProcessInstance request);

	/**
	 * delete process instance by id
	 *
	 * @param processInstanceId
	 * @return
	 */
	ProcessInstance deleteProcessInstance(String processInstanceId);

	/**
	 * list all the process histories
	 *
	 * @param processInstanceId
	 * @return
	 */
	List<ProcessHistory> listProcessHistories(String processInstanceId);

	/**
	 * find process history by id
	 *
	 * @param historyId
	 * @return
	 */
	ProcessHistory findProcessHistoryById(String historyId);

	/**
	 * save the process history
	 *
	 * @param request
	 * @return
	 */
	ProcessHistory saveProcessHistory(ProcessHistory request);

	/**
	 * save the activity instance
	 *
	 * @param request
	 * @return
	 */
	ActivityInstance saveActivityInstance(ActivityInstance request);

	/**
	 * update the status of activity instance
	 *
	 * @param activityInstanceId
	 * @param status
	 * @return
	 */
	ActivityInstance updateActivityInstance(String activityInstanceId, ActivityStatus status);

	/**
	 * find activity instance by id
	 *
	 * @param activityInstanceId
	 * @return
	 */
	ActivityInstance findActivityInstanceById(String activityInstanceId);

	/**
	 * dynamic query activity instance
	 *
	 * @param queryRequest
	 * @return
	 */
	Page<ActivityInstance> listActivityInstances(ActivityQueryRequest queryRequest);

	/**
	 * list all the activity instances which is type of specified activity definition
	 *
	 * @param activityDefinition
	 * @param businessReference
	 * @return
	 */
	List<ActivityInstance> listActivityInstances(ActivityDefinition activityDefinition,
												 BusinessReference businessReference);

	/**
	 * find the activity instance by definition and receiver
	 *
	 * @param processInstance
	 * @param nextActivityDefinition
	 * @param receiver
	 * @return
	 */
	ActivityInstance findActivityInstance(ProcessInstance processInstance,
										  ActivityDefinition nextActivityDefinition,
										  UserReference receiver);
}
