package in.clouthink.daas.sbpm.core.spi.impl;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import in.clouthink.daas.sbpm.core.exception.ProcessDefinitionException;
import in.clouthink.daas.sbpm.core.exception.ProcessDefinitionParserException;
import in.clouthink.daas.sbpm.core.exception.WorkflowException;
import in.clouthink.daas.sbpm.core.model.ActivityDefinition;
import in.clouthink.daas.sbpm.core.model.ActivityType;
import in.clouthink.daas.sbpm.core.model.DefaultActivityDefinition;
import in.clouthink.daas.sbpm.core.spi.ProcessDefinitionParser;
import org.springframework.util.StringUtils;

import java.io.*;
import java.util.*;

/**
 * @author dz
 */
public class ProcessDefinitionJsonParser implements ProcessDefinitionParser {

	ObjectMapper objectMapper = new ObjectMapper();

	@Override
	public List<ActivityDefinition> parse(InputStream inputStream) {
		try {
			List<ActivityDefinition> result = objectMapper.readValue(inputStream,
																	 new TypeReference<List<DefaultActivityDefinition>>() {
																	 });
			validate(result);
			return result;
		}
		catch (WorkflowException e) {
			throw e;
		}
		catch (Exception e) {
			throw new ProcessDefinitionParserException("解析内容出错", e);
		}
	}

	@Override
	public List<ActivityDefinition> parse(File file) {
		try {
			List<ActivityDefinition> result = parse(new FileInputStream(file));
			validate(result);
			return result;
		}
		catch (WorkflowException e) {
			throw e;
		}
		catch (IOException e) {
			throw new ProcessDefinitionParserException(String.format("解析文件'%s'出错", file.getName()), e);
		}
	}

	@Override
	public List<ActivityDefinition> parse(String content) {
		return parse(new ByteArrayInputStream(content.getBytes()));
	}

	@Override
	public String toString(List<ActivityDefinition> activityDefinitions) {
		try {
			return objectMapper.writeValueAsString(activityDefinitions);
		}
		catch (JsonProcessingException e) {
			throw new ProcessDefinitionParserException("保存流程定义出错", e);
		}
	}

	private void validate(List<ActivityDefinition> activityDefinitionList) {
		Map<String,ActivityDefinition> idValueMap = new HashMap<>();
		Set<String> idSet = new HashSet<>();

		int startCounter = 0;
		for (ActivityDefinition item : activityDefinitionList) {
			if (item.getType() == ActivityType.Start) {
				startCounter++;
			}
			idValueMap.put(item.getId(), item);
			idSet.add(item.getId());
		}
		if (startCounter == 0) {
			throw new ProcessDefinitionException("流程定义错误:没有定义开始节点");
		}
		if (startCounter > 1) {
			throw new ProcessDefinitionException("流程定义错误:定义了多个开始节点");
		}
		if (idSet.size() < activityDefinitionList.size()) {
			throw new ProcessDefinitionException("流程定义错误:流程活动节点id重复");
		}

		activityDefinitionList.stream().forEach(defItem -> {
			if (StringUtils.isEmpty(defItem.getId())) {
				throw new ProcessDefinitionException("流程定义错误:流程活动节点id不能为空");
			}
			if (StringUtils.isEmpty(defItem.getName())) {
				throw new ProcessDefinitionException("流程定义错误:流程活动节点name不能为空");
			}
			if (defItem.getType() == null) {
				throw new ProcessDefinitionException("流程定义错误:流程活动节点type不能为空");
			}
			if (defItem.getRouters() == null || defItem.getRouters().isEmpty()) {
				if (defItem.getType() != ActivityType.End) {
					throw new ProcessDefinitionException("流程定义错误:流程活动节点router不能为空");
				}
			}

			Set<String> routerCodeSet = new HashSet<>();
			defItem.getRouters().stream().forEach(routerItem -> {
				if (StringUtils.isEmpty(routerItem.getCode())) {
					throw new ProcessDefinitionException("流程定义错误:路由编码不能为空");
				}
				if (StringUtils.isEmpty(routerItem.getName())) {
					throw new ProcessDefinitionException("流程定义错误:路由名称不能为空");
				}
				if ("process".equals(routerItem.getType()) && StringUtils.isEmpty(routerItem.getNextActivityId())) {
					throw new ProcessDefinitionException("流程定义错误:路由目标（流转到下一活动节点）不能为空");
				}
				if ("process".equals(routerItem.getType()) && idValueMap.get(routerItem.getNextActivityId()) == null) {
					throw new ProcessDefinitionException("流程定义错误:路由目标（流转到下一活动节点）对应的活动未定义");
				}
				if ("complete".equals(routerItem.getType()) && defItem.getType() != ActivityType.End) {
					throw new ProcessDefinitionException("流程定义错误:只有结束节点能定义complete类型的路由");
				}

				routerCodeSet.add(routerItem.getCode());
			});

			if (routerCodeSet.size() < defItem.getRouters().size()) {
				throw new ProcessDefinitionException("流程定义错误:路由编码不能重复");
			}
		});

	}

}
