package in.clouthink.daas.sbpm.mongodb.spiImpl;

import in.clouthink.daas.sbpm.core.exception.*;
import in.clouthink.daas.sbpm.core.model.*;
import in.clouthink.daas.sbpm.core.service.AbstractWorkflowService;
import in.clouthink.daas.sbpm.core.spi.ParticipantParser;
import in.clouthink.daas.sbpm.mongodb.model.ProcessHistory;
import in.clouthink.daas.sbpm.mongodb.model.ProcessInstance;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.StringUtils;

import java.util.Date;
import java.util.List;

/**
 * The workflow service impl.
 *
 * @author dz
 */
public class WorkflowServiceImpl extends AbstractWorkflowService {

	@Autowired
	private ParticipantParser participantParser;

	public ParticipantParser getParticipantParser() {
		return participantParser;
	}

	public void setParticipantParser(ParticipantParser participantParser) {
		this.participantParser = participantParser;
	}

	@Override
	public ProcessInstance startProcess(String processDefinitionId, ActionRequest actionRequest, UserReference user) {
		if (StringUtils.isEmpty(processDefinitionId)) {
			throw new ProcessDefinitionException("请指定流程定义id");
		}

		ProcessDefinition processDefinition = getWorkflowRepository().findProcessDefinitionById(processDefinitionId);
		if (processDefinition == null) {
			throw new ProcessDefinitionNotFoundException("无效流程定义id");
		}

		ActivityDefinition startActivityDefinition = getStartActivityDefinition(processDefinitionId);
		if (startActivityDefinition == null) {
			throw new ActivityDefinitionNotFoundException("流程定义错误:没有定义开始节点");
		}

		ActivityRouter startRouter = startActivityDefinition.getRouters().get(0);
		if (startRouter == null) {
			throw new ActivityDefinitionException("流程定义错误:流程开始活动节点没有定义流转信息");
		}

		ActivityDefinition nextActivityDefinition = getWorkflowRepository().findActivityDefinitionById(startRouter.getNextActivityId());
		if (nextActivityDefinition == null) {
			throw new ActivityDefinitionNotFoundException("流程定义错误:没有找到流转的下一节点或者流转的下一节点无效");
		}

		String participantExpr = nextActivityDefinition.getParticipant();
		List<Participant> sysUserList = participantParser.parse(participantExpr);
		if (sysUserList == null || sysUserList.isEmpty()) {
			throw new ActivityDefinitionException("流程定义错误:流转的下一活动节点未定义参与者");
		}

		if (StringUtils.isEmpty(actionRequest.getBizRefId())) {
			throw new ProcessInstanceException("业务id不能为空");
		}
		if (StringUtils.isEmpty(actionRequest.getBizType())) {
			throw new ProcessInstanceException("业务类型不能为空");
		}

		DefaultNamedBusinessReference businessReference = new DefaultNamedBusinessReference();
		businessReference.setRefId(actionRequest.getBizRefId());
		businessReference.setType(actionRequest.getBizType());
		businessReference.setName(actionRequest.getBizName());

		//after validate, do workflow engine logic
		DefaultNamedUserReference userReference = DefaultNamedUserReference.from(user);

		ProcessInstance processInstance = new ProcessInstance();
		processInstance.setProcessDefinition(DefaultProcessDefinitionReference.from(processDefinition));
		processInstance.setBusinessReference(businessReference);
		processInstance.setCreatedAt(new Date());
		processInstance.setCreatedBy(userReference);
		processInstance.setStatus(ProcessStatus.Processing);

		processInstance = (ProcessInstance) getWorkflowRepository().saveProcessInstance(processInstance);

		//handle the start process history
		ProcessHistory processHistory = new ProcessHistory();
		processHistory.setName("开始");
		processHistory.setProcessBy(userReference);
		processHistory.setProcessAt(new Date());
		processHistory.setAction(DefaultActionReference.StartProcessActionReference);
		processHistory.setActivityInstance(null); //start the process (no activity instance found)
		processHistory.setMemo(null);
		processHistory.setProcessInstance(processInstance);
		processHistory = (ProcessHistory) getWorkflowRepository().saveProcessHistory(processHistory);

		//create the start activity instance and mark it as process automatically
		DefaultActionReference startAction = new DefaultActionReference();
		startAction.setCode(startRouter.getCode());
		startAction.setName(startRouter.getName());

		in.clouthink.daas.sbpm.mongodb.model.ActivityInstance startActivityInstance = new in.clouthink.daas.sbpm.mongodb.model.ActivityInstance();
		startActivityInstance.setName(startActivityDefinition.getName());
		startActivityInstance.setProcessInstance(processInstance);
		startActivityInstance.setBusinessReference(businessReference);
		startActivityInstance.setActivityDefinition((in.clouthink.daas.sbpm.mongodb.model.ActivityDefinition) startActivityDefinition);
		startActivityInstance.setFromAction(DefaultActionReference.StartProcessActionReference);
		startActivityInstance.setFromBy(userReference);
		startActivityInstance.setReceiveAt(new Date());
		startActivityInstance.setReceiver(userReference);
		startActivityInstance.setProcessAction(startAction);
		startActivityInstance.setProcessBy(userReference);
		startActivityInstance.setProcessAt(new Date());
		startActivityInstance.setStatus(ActivityStatus.Processed);

		startActivityInstance = (in.clouthink.daas.sbpm.mongodb.model.ActivityInstance) getWorkflowRepository().saveActivityInstance(
				startActivityInstance);

		processHistory = new ProcessHistory();
		processHistory.setName(startActivityInstance.getName());
		processHistory.setProcessBy(userReference);
		processHistory.setProcessAt(new Date());
		processHistory.setAction(startAction);
		processHistory.setActivityInstance(startActivityInstance); //start the process (no activity instance found)
		processHistory.setMemo(actionRequest.getMemo());
		processHistory.setProcessInstance(processInstance);
		processHistory = (ProcessHistory) getWorkflowRepository().saveProcessHistory(processHistory);

		//now do the action on the start activity to move to next activity
		for (Participant participantUser : sysUserList) {
			DefaultNamedUserReference receiverReference = new DefaultNamedUserReference();
			receiverReference.setRefId(participantUser.getId());
			receiverReference.setType(participantUser.getType());
			receiverReference.setName(participantUser.getUsername());

			in.clouthink.daas.sbpm.mongodb.model.ActivityInstance nextActivityInstance = new in.clouthink.daas.sbpm.mongodb.model.ActivityInstance();
			nextActivityInstance.setName(nextActivityDefinition.getName());
			nextActivityInstance.setProcessInstance(processInstance);
			nextActivityInstance.setBusinessReference(businessReference);
			nextActivityInstance.setActivityDefinition((in.clouthink.daas.sbpm.mongodb.model.ActivityDefinition) nextActivityDefinition);
			nextActivityInstance.setFromAction(startAction);
			nextActivityInstance.setFromBy(userReference);
			nextActivityInstance.setReceiveAt(new Date());
			nextActivityInstance.setReceiver(receiverReference);
			nextActivityInstance.setStatus(ActivityStatus.Pending);

			nextActivityInstance = (in.clouthink.daas.sbpm.mongodb.model.ActivityInstance) getWorkflowRepository().saveActivityInstance(
					nextActivityInstance);
		}

		return processInstance;
	}

	@Override
	public in.clouthink.daas.sbpm.mongodb.model.ActivityInstance claimActivity(String activityInstanceId,
																			   UserReference user) {
		in.clouthink.daas.sbpm.mongodb.model.ActivityInstance activityInstance = (in.clouthink.daas.sbpm.mongodb.model.ActivityInstance) getWorkflowRepository()
				.findActivityInstanceById(activityInstanceId);
		if (activityInstance == null) {
			throw new ActivityInstanceNotFoundException();
		}

		//when the activity instance is pending status, do the claim logic
		if (activityInstance.getStatus() == ActivityStatus.Processed) {
			throw new ProcessInstanceException("该任务已经处理,请勿重复签收");
		}

		if (activityInstance.getStatus() == ActivityStatus.ByOther) {
			throw new ProcessInstanceException("该任务已经被其他人抢先处理,请刷新后重试");
		}

		if (activityInstance.getStatus() == ActivityStatus.Claimed) {
			throw new ProcessInstanceException("该任务已签收,请勿重复签收");
		}

		if (activityInstance.getStatus() == ActivityStatus.Pending) {
			//如果是抢占式任务,需要处理其他参与者的任务状态
			List<ActivityInstance> activityInstanceList = getWorkflowRepository().listActivityInstances(activityInstance
																												.getActivityDefinition(),
																										activityInstance
																												.getBusinessReference());
			activityInstanceList.stream().forEach(item -> {
				in.clouthink.daas.sbpm.mongodb.model.ActivityInstance otherActivityInstance = (in.clouthink.daas.sbpm.mongodb.model.ActivityInstance) item;
				if (!otherActivityInstance.getId().equals(activityInstanceId)) {
					otherActivityInstance.setStatus(ActivityStatus.ByOther);
					otherActivityInstance = (in.clouthink.daas.sbpm.mongodb.model.ActivityInstance) getWorkflowRepository()
							.saveActivityInstance(otherActivityInstance);
				}
			});

			//签收任务
			activityInstance.setStatus(ActivityStatus.Claimed);
			activityInstance = (in.clouthink.daas.sbpm.mongodb.model.ActivityInstance) getWorkflowRepository().saveActivityInstance(
					activityInstance);
		}

		return activityInstance;
	}

	@Override
	public in.clouthink.daas.sbpm.mongodb.model.ActivityInstance processActivity(String activityInstanceId,
																				 ActionRequest actionRequest,
																				 UserReference user) {
		if (StringUtils.isEmpty(actionRequest.getAction())) {
			throw new ProcessInstanceException("请指定流转动作");
		}

		in.clouthink.daas.sbpm.mongodb.model.ActivityInstance activityInstance = (in.clouthink.daas.sbpm.mongodb.model.ActivityInstance) getWorkflowRepository()
				.findActivityInstanceById(activityInstanceId);
		if (activityInstance == null) {
			throw new ActivityInstanceNotFoundException();
		}

		if (activityInstance.getStatus() == ActivityStatus.Processed) {
			throw new ProcessInstanceException("该任务已经处理,请勿重复处理");
		}

		if (activityInstance.getStatus() == ActivityStatus.ByOther) {
			throw new ProcessInstanceException("该任务已经被其他人抢先处理");
		}

		if (activityInstance.getStatus() != ActivityStatus.Claimed) {
			throw new ProcessInstanceException("该任务未签收,请先签收再处理");
		}

		ProcessInstance processInstance = activityInstance.getProcessInstance();
		if (processInstance.getStatus() == ProcessStatus.Complete) {
			throw new ProcessInstanceException("该流程已结束");
		}

		ActivityDefinition activityDefinition = activityInstance.getActivityDefinition();

		List<? extends ActivityRouter> activityRouterList = activityDefinition.getRouters();
		ActivityRouter matchedRouter = null;
		for (ActivityRouter router : activityRouterList) {
			if (router.getCode().equalsIgnoreCase(actionRequest.getAction())) {
				matchedRouter = router;
				break;
			}
		}

		if (matchedRouter == null) {
			throw new ProcessInstanceException("无效的流转动作");
		}

		if (matchedRouter.isMemoRequired() && StringUtils.isEmpty(actionRequest.getMemo())) {
			throw new ProcessInstanceException("请提供处理意见");
		}


		//after validate, do workflow engine logic
		DefaultNamedUserReference userReference = DefaultNamedUserReference.from(user);

		DefaultActionReference currentAction = new DefaultActionReference();
		currentAction.setCode(matchedRouter.getCode());
		currentAction.setName(matchedRouter.getName());

		if ("complete".equalsIgnoreCase(matchedRouter.getType())) {
			processInstance.setStatus(ProcessStatus.Complete);
			getWorkflowRepository().updateProcessInstance(processInstance.getId(), ProcessStatus.Complete);

			activityInstance.setStatus(ActivityStatus.Processed);
			activityInstance.setProcessAt(new Date());
			activityInstance.setProcessBy(userReference);
			activityInstance.setProcessAction(currentAction);

			activityInstance = (in.clouthink.daas.sbpm.mongodb.model.ActivityInstance) getWorkflowRepository().saveActivityInstance(
					activityInstance);

			//handle the process history
			ProcessHistory processHistory = new ProcessHistory();
			processHistory.setName(activityInstance.getName());
			processHistory.setProcessBy(userReference);
			processHistory.setProcessAt(new Date());
			processHistory.setAction(currentAction);
			processHistory.setActivityInstance(activityInstance); //start the process (no activity instance found)
			processHistory.setMemo(actionRequest.getMemo());
			processHistory.setProcessInstance(activityInstance.getProcessInstance());
			processHistory = (ProcessHistory) getWorkflowRepository().saveProcessHistory(processHistory);

			return activityInstance;
		}

		ActivityDefinition nextActivityDefinition = getWorkflowRepository().findActivityDefinitionById(matchedRouter.getNextActivityId());
		if (nextActivityDefinition == null) {
			throw new ActivityDefinitionNotFoundException("流程定义错误:没有指定下一个活动节点或者定义的下一个活动节点无效");
		}

		if (nextActivityDefinition.getType() == ActivityType.Start) {
			//如果指定的下一个节点是开始节点,则自动分配节点的用户(即启动流程的用户)
			UserReference startUser = activityInstance.getProcessInstance().getCreatedBy();
			in.clouthink.daas.sbpm.mongodb.model.ActivityInstance startActivityInstance = (in.clouthink.daas.sbpm.mongodb.model.ActivityInstance) getWorkflowRepository()
					.findActivityInstance(processInstance, nextActivityDefinition, startUser);
			if (startActivityInstance == null) {
				throw new ProcessInstanceException("流程数据错误:当前操作将流转到流程开始活动节点,但是对应的活动实例不存在");
			}

			//reset the activity instance's status to pending instead of to create new one
			startActivityInstance.setFromAction(currentAction);
			startActivityInstance.setFromBy(userReference);
			startActivityInstance.setReceiveAt(new Date());
			startActivityInstance.setStatus(ActivityStatus.Pending);
			startActivityInstance.setProcessAction(null);
			startActivityInstance.setProcessAt(null);
			startActivityInstance.setProcessBy(null);

			startActivityInstance = (in.clouthink.daas.sbpm.mongodb.model.ActivityInstance) getWorkflowRepository().saveActivityInstance(
					startActivityInstance);
		}
		else {
			String participantExpr = nextActivityDefinition.getParticipant();
			List<Participant> sysUserList = participantParser.parse(participantExpr);
			if (sysUserList == null || sysUserList.isEmpty()) {
				throw new ActivityDefinitionNotFoundException("流程定义错误:下一个活动节点没有定义参与者");
			}

			//now do the action on current activity to move to next activity
			for (Participant participantUser : sysUserList) {
				DefaultNamedUserReference receiverReference = new DefaultNamedUserReference();
				receiverReference.setRefId(participantUser.getId());
				receiverReference.setType(participantUser.getType());
				receiverReference.setName(participantUser.getUsername());

				//handle the re-loop logic
				in.clouthink.daas.sbpm.mongodb.model.ActivityInstance nextActivityInstance = (in.clouthink.daas.sbpm.mongodb.model.ActivityInstance) getWorkflowRepository()
						.findActivityInstance(processInstance, nextActivityDefinition, receiverReference);
				if (nextActivityInstance != null) {
					//reset the activity instance's status to pending instead of to create new one
					nextActivityInstance.setFromAction(currentAction);
					nextActivityInstance.setFromBy(userReference);
					nextActivityInstance.setReceiveAt(new Date());
					nextActivityInstance.setStatus(ActivityStatus.Pending);
					nextActivityInstance.setProcessAction(null);
					nextActivityInstance.setProcessAt(null);
					nextActivityInstance.setProcessBy(null);

					nextActivityInstance = (in.clouthink.daas.sbpm.mongodb.model.ActivityInstance) getWorkflowRepository()
							.saveActivityInstance(nextActivityInstance);
				}
				else {
					nextActivityInstance = new in.clouthink.daas.sbpm.mongodb.model.ActivityInstance();
					nextActivityInstance.setName(nextActivityDefinition.getName());
					nextActivityInstance.setBusinessReference(activityInstance.getBusinessReference());
					nextActivityInstance.setProcessInstance(activityInstance.getProcessInstance());
					nextActivityInstance.setActivityDefinition((in.clouthink.daas.sbpm.mongodb.model.ActivityDefinition) nextActivityDefinition);
					nextActivityInstance.setFromAction(currentAction);
					nextActivityInstance.setFromBy(userReference);
					nextActivityInstance.setReceiveAt(new Date());
					nextActivityInstance.setReceiver(receiverReference);
					nextActivityInstance.setStatus(ActivityStatus.Pending);

					nextActivityInstance = (in.clouthink.daas.sbpm.mongodb.model.ActivityInstance) getWorkflowRepository()
							.saveActivityInstance(nextActivityInstance);
				}
			}
		}

		activityInstance.setStatus(ActivityStatus.Processed);
		activityInstance.setProcessAt(new Date());
		activityInstance.setProcessBy(userReference);
		activityInstance.setProcessAction(currentAction);

		activityInstance = (in.clouthink.daas.sbpm.mongodb.model.ActivityInstance) getWorkflowRepository().saveActivityInstance(
				activityInstance);

		//handle the process history
		ProcessHistory processHistory = new ProcessHistory();
		processHistory.setName(activityInstance.getName());
		processHistory.setProcessBy(userReference);
		processHistory.setProcessAt(new Date());
		processHistory.setAction(currentAction);
		processHistory.setActivityInstance(activityInstance); //start the process (no activity instance found)
		processHistory.setMemo(actionRequest.getMemo());
		processHistory.setProcessInstance(activityInstance.getProcessInstance());
		processHistory = (ProcessHistory) getWorkflowRepository().saveProcessHistory(processHistory);

		return activityInstance;
	}

}
