package com.dilwar.bindingAdapters

import com.dilwar.GRecyclerFilterListener
import android.view.LayoutInflater
import android.view.ViewGroup
import android.widget.Filter
import android.widget.Filterable
import androidx.annotation.LayoutRes
import androidx.databinding.DataBindingUtil
import androidx.databinding.ViewDataBinding
import androidx.recyclerview.widget.RecyclerView

/**
 * GRecyclerBindingAdapter
 *
 * @param M Data-Model to be used in RecyclerView Adapter
 * @param B DataBindingClass Name For the Layout
 *
 * @property layoutRes Layout Resource from Displaying RecyclerView
 */
//(@LayoutRes val layoutRes: Int)
class GRecyclerBindingAdapter<M, B : ViewDataBinding>
private constructor(@LayoutRes val layoutRes: Int) :
    RecyclerView.Adapter<GRecyclerBindingAdapter.ViewHolder<B>>(),
    Filterable {

    //   var populateListener: GRecyclerBindingListener<M, B>? = null
    private val primaryDataList = mutableListOf<M>()
    private val filterList = mutableListOf<M>()
    private var populateListener: GRecyclerBindingListener<M, B>? = null
    private var filterListener: GRecyclerFilterListener<M>? = null
    private var br: Int? = null

    constructor(
        @LayoutRes layoutRes: Int,
        populateListener: GRecyclerBindingListener<M, B>?,
        filterListener: GRecyclerFilterListener<M>
    ) : this(layoutRes) {
        this.populateListener = populateListener
        this.filterListener = filterListener
    }

    constructor(
        @LayoutRes layoutRes: Int,
        populateListener: GRecyclerBindingListener<M, B>?
    ) : this(layoutRes) {
        this.populateListener = populateListener
    }

    constructor(
        @LayoutRes layoutRes: Int,
        br: Int,
        filterListener: GRecyclerFilterListener<M>
    ) : this(layoutRes) {
        this.br = br
        this.filterListener = filterListener
    }

    constructor(
        @LayoutRes layoutRes: Int,
        br: Int
    ) : this(layoutRes) {
        this.br = br
    }

    init {
        filterListener = GRecyclerFilterListener.getActualOrDefault(filterListener)
    }

    /**
     * PopulateRecyclerView
     *
     * @param list List of Data-Model
     * @return
     */
    fun submitList(list: List<M>?): GRecyclerBindingAdapter<M, B> {
        primaryDataList.clear()
        primaryDataList.addAll(list ?: emptyList())

        filterList.clear()
        filterList.addAll(primaryDataList)

        notifyDataSetChanged()
        return this
    }

    fun getAllItems(): List<M> {
        return primaryDataList
    }

    fun getFilteredList(): List<M> {
        return filterList
    }

    fun addItem(item: M) {
        addItem(item, filterList.size)
    }

    fun addItem(item: M, position: Int) {
        primaryDataList.add(position, item)
        filterList.add(position, item)
        notifyItemInserted(position)
    }

    fun removeItemAt(position: Int) {
        val data = filterList[position]
        removeItem(data)
    }

    fun removeItem(data: M) {
        primaryDataList.remove(data)
        filterList.remove(data)
        notifyDataSetChanged()
    }

    /**
     * onCreateViewHolder
     *
     * @param parent ViewGroup
     * @param viewType Integer
     * @return
     */
    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ViewHolder<B> {
        val bindingUtil = DataBindingUtil
            .inflate<B>(LayoutInflater.from(parent.context), layoutRes, parent, false)
        return ViewHolder(bindingUtil)
    }

    override fun getItemCount() = filterList.size

    /**
     * onBindViewHolder
     *
     * @param holder viewholder
     * @param position postion of view
     */
    override fun onBindViewHolder(holder: ViewHolder<B>, position: Int) {
        if (populateListener != null) {
            populateListener!!.populateItemBindingHolder(holder, filterList[position], position)
        } else {
            holder.binding.setVariable(br!!, filterList[position])
        }
        holder.binding.executePendingBindings()
    }

    /**
     * ViewHolder For RecyclerView
     *
     * @param B Data Binding Class
     * @property b is the type DataBinding Class
     */
    class ViewHolder<B : ViewDataBinding>(val binding: B) : RecyclerView.ViewHolder(binding.root)


    override fun getFilter(): Filter {
        return object : Filter() {
            override fun performFiltering(constraint: CharSequence?): FilterResults {
                val tempList = emptyList<M>().toMutableList()

                if (constraint.toString().isEmpty()) {
                    tempList.addAll(primaryDataList)
                } else {
                    primaryDataList.forEach {
                        if (filterListener != null) {
                            if (filterListener!!.itemFilter(constraint.toString(), it))
                                tempList.add(it)
                        }
                    }
                }

                val filterResult = FilterResults()
                filterResult.values = tempList
                filterResult.count = tempList.size

                return filterResult
            }


            override fun publishResults(constraint: CharSequence?, results: FilterResults?) {
                val list = results?.values as? List<M> ?: emptyList()
                filterList.clear()
                filterList.addAll(list)
                notifyDataSetChanged()
            }

        }
    }

}
