package in.juspay.model;

import com.nimbusds.jose.crypto.bc.BouncyCastleProviderSingleton;
import in.juspay.exception.InvalidKeysException;
import in.juspay.exception.JuspayException;

import javax.xml.bind.DatatypeConverter;
import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

public class JweJwsEncryptionKeys {
    private PublicKey jwePublicKey;
    private PrivateKey jwsPrivateKey;
    private String keyId;

    public JweJwsEncryptionKeys(String keyId, String publicKey, String privateKey) throws JuspayException {
        this.withKeyId(keyId);
        this.withJwePublicKey(publicKey);
        this.withJwsPrivateKey(privateKey);
    }

    static {
        Provider bouncyCastleProvider = BouncyCastleProviderSingleton.getInstance();
        Security.addProvider(bouncyCastleProvider);
    }

    public void withKeyId(String val) {
        keyId = val;
    }

    public void withJwsPrivateKey(String key) throws JuspayException {
        try {
            KeyFactory kf = KeyFactory.getInstance("RSA");
            String keyContent = sanitizePrivateKey(key);
            PKCS8EncodedKeySpec keySpecPv = new PKCS8EncodedKeySpec(DatatypeConverter.parseBase64Binary(keyContent));
            jwsPrivateKey = kf.generatePrivate(keySpecPv);
        } catch (Exception e) {
            throw new InvalidKeysException(-1, "invalid_jws_private_key", "invalid_jws_private_key", "Please check your private key");
        }
    }


    public void withJwePublicKey(String key) throws JuspayException {
        try {
            KeyFactory kf = KeyFactory.getInstance("RSA");
            byte[] pokey = DatatypeConverter.parseBase64Binary(sanitizePublicKey(key));
            jwePublicKey = kf.generatePublic(new X509EncodedKeySpec(pokey));
        } catch (Exception e) {
            throw new InvalidKeysException(-1, "invalid_jwe_public_key", "invalid_jwe_public_key", "Please check your public key");
        }
    }

    public PrivateKey getJwsPrivateKey() {
        return jwsPrivateKey;
    }

    public PublicKey getJwePublicKey() {
        return jwePublicKey;
    }

    public String getKeyId() {
        return keyId;
    }

    protected String sanitizePrivateKey(String key){
        return key.replaceAll("\\n", "")
                .replace("-----BEGIN PRIVATE KEY-----", "")
                .replace("-----END PRIVATE KEY-----", "")
                .replace("-----BEGIN RSA PRIVATE KEY-----", "")
                .replace("-----END RSA PRIVATE KEY-----", "")
                .replace("\r\n","");
    }

    protected String sanitizePublicKey(String key){
        return key.replaceAll("\\n", "")
                .replace("-----BEGIN PUBLIC KEY-----", "")
                .replace("-----END PUBLIC KEY-----", "")
                .replace("-----BEGIN RSA PUBLIC KEY-----", "")
                .replace("-----END RSA PUBLIC KEY-----", "")
                .replace("\r\n", "");
    }
}
