package in.juspay.security;

import java.security.*;
import java.security.interfaces.RSAPrivateKey;

import java.util.LinkedHashMap;
import java.util.Map;

public class JWS {
    final public static String alg = "RS256";

    public static String sign(String claims, String keyId, PrivateKey privateKey) throws JuspayCryptoException {

        if (!(privateKey instanceof RSAPrivateKey)) {
            throw new JuspayCryptoException("private key algorithm is not RSA");
        }

        if (claims == null || keyId == null) {
            throw new JuspayCryptoException("claims and keyId cannot be null");
        }

        String signatureHeader = "{\"alg\":\"RS256\",\"kid\":\"" + keyId + "\"}";
        String header = Utils.base64UrlEncodeAsString(signatureHeader);
        String payload = Utils.base64UrlEncodeAsString(claims);
        String data = header + "." + payload;

        Signature signatureSPI;
        try {
            signatureSPI = Signature.getInstance("SHA256withRSA");
        } catch (NoSuchAlgorithmException e) {
            throw new JuspayCryptoException("No such algorithm found:- " + e.getMessage(), e);
        }

        try {
            signatureSPI.initSign(privateKey);
        } catch (InvalidKeyException e) {
            throw new JuspayCryptoException("Invalid private key:- " + e.getMessage(), e);
        }

        byte[] signatureBytes;
        try {
            signatureSPI.update(data.getBytes());
            signatureBytes = signatureSPI.sign();
        } catch (SignatureException e) {
            throw new JuspayCryptoException("Signature Exception:- " + e.getMessage(), e);
        }

        String jwsSignature = Utils.base64UrlEncodeAsString(signatureBytes);
        return header + "." + payload + "." + jwsSignature;
    }

    public static String verify(Map<String, String> signedPayload, PublicKey publicKey) throws JuspayCryptoException {
        String header = signedPayload.get("header");
        String signature = signedPayload.get("signature");
        String payload = signedPayload.get("payload");

        String dataToVerify = header + "." + payload;
        Signature signatureSPI;
        try {
            signatureSPI = Signature.getInstance("SHA256withRSA");
        } catch (NoSuchAlgorithmException e) {
            throw new JuspayCryptoException("No such algorithm found:- " + e.getMessage(), e);
        }

        try {
            signatureSPI.initVerify(publicKey);
        } catch (InvalidKeyException e) {
            throw new JuspayCryptoException("Invalid private key:- " + e.getMessage(), e);
        }

        try {
            signatureSPI.update(dataToVerify.getBytes());
            signatureSPI.verify(Utils.base64UrlDecode(signature));
        } catch (SignatureException e) {
            throw new JuspayCryptoException("Signature Exception:- " + e.getMessage(), e);
        }

        return Utils.base64UrlDecodeAsString(payload);
    }

    public static String verify(String signedPayload, PublicKey publicKey) throws JuspayCryptoException {
        String[] encryptedPayloadParts = signedPayload.split("\\.");
        Map<String, String> reqASMap = new LinkedHashMap<String, String>();
        if (encryptedPayloadParts.length != 3) {
            throw new JuspayCryptoException("Request payload malformed");
        }
        reqASMap.put("header",encryptedPayloadParts[0]);
        reqASMap.put("payload",encryptedPayloadParts[1]);
        reqASMap.put("signature",encryptedPayloadParts[2]);
        return verify(reqASMap, publicKey);
    }
}
