package in.juspay.model;

import com.google.gson.*;
import com.google.gson.reflect.TypeToken;
import in.juspay.exception.*;
import in.juspay.security.JWE;
import in.juspay.security.JWS;
import in.juspay.security.JuspayCryptoException;
import org.apache.commons.codec.binary.Base64;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.MapMessage;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import java.io.*;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.net.URLEncoder;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.util.*;

public abstract class JuspayEntity {

    private static final Logger log = LogManager.getLogger(JuspayEntity.class);

    private static String serializeParams(Map<String, Object> params) {
        if (params == null || params.size() == 0) {
            return "";
        }
        StringBuilder serializedParams = new StringBuilder();
        try {
            for (String key : params.keySet()) {
                serializedParams.append(key + "=");
                if (params.get(key) != null) {
                    serializedParams.append(URLEncoder.encode(params.get(key).toString(), "UTF-8"));
                }
                serializedParams.append("&");
            }
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Encoding exception while trying to construct payload", e);
        }
        if (serializedParams.charAt(serializedParams.length() - 1) == '&') {
            serializedParams.deleteCharAt(serializedParams.length() - 1);
        }
        return serializedParams.toString();
    }
    private static String prepareParam(Map<String, Object> params, String contentType) {
        String res = "";
        if(contentType == "application/json") {
            Gson gson = new Gson();
            res = gson.toJson(params).toString();
        } else {
            res = serializeParams(params);
        }
        return res;
    }

    protected static AuthMethod getEncryptionMethodBasedOnConfig(RequestOptions requestOptions) {
        if(requestOptions == null) {
            requestOptions = RequestOptions.createDefault();
        }
        return requestOptions.getJweJwsEncryptionKeys() == null ? AuthMethod.BASIC : AuthMethod.JWE_JWS;
    }

    protected static JsonObject makeServiceCall(String path, Map<String, Object> params, RequestMethod method, RequestOptions requestOptions)
            throws APIException, APIConnectionException, AuthorizationException, AuthenticationException, InvalidRequestException {
        return makeServiceCall(path, params, method, requestOptions, AuthMethod.BASIC);
    }

    protected static JsonObject makeServiceCallWithAuthDecider(String path, Map<String, Object> params, RequestMethod method, RequestOptions requestOptions)
            throws APIException, APIConnectionException, AuthorizationException, AuthenticationException, InvalidRequestException {
        JsonObject response =  null;
        if(getEncryptionMethodBasedOnConfig(requestOptions) == AuthMethod.JWE_JWS) {
            response = makeServiceCall(path, params, RequestMethod.POST, requestOptions, AuthMethod.JWE_JWS);
        } else {
            response = makeServiceCall(path, params, method, requestOptions, AuthMethod.BASIC);
        }
        return response;
    }

    protected static JsonObject makeServiceCall(String path, Map<String, Object> params, RequestMethod method, RequestOptions requestOptions, AuthMethod authMethod)
            throws APIException, APIConnectionException, AuthorizationException, AuthenticationException, InvalidRequestException {
        if (requestOptions == null) {
            requestOptions = RequestOptions.createDefault();
        }

        int httpResponseCode = -1;
        String responseString = null;
        HttpsURLConnection con = null;
        InputStream inputStream = null;
        BufferedReader reader = null;
        try {

            switch (authMethod) {
                case JWE_JWS:
                    JweJwsEncryptionKeys jweJwsEncryptionKeys = requestOptions.getJweJwsEncryptionKeys();
                    if(jweJwsEncryptionKeys == null) {
                        throw new InvalidKeysException(-1, "invalid.auth", "invalid.auth", "JWE JWS encryption keys are not configured.");
                    }
                    params = encryptRequest(params, jweJwsEncryptionKeys);
                    break;
                case BASIC:
                default:
                    break;
            }

            MapMessage mapMessage = new MapMessage();
            if (params != null) {
                for (String key : params.keySet()) {
                    String value = "";
                    if (params.get(key) != null) {
                        value = params.get(key).toString();
                    }
                    mapMessage.put(key, value);
                }
            }

            String baseUrl = requestOptions.getBaseUrl() == null || requestOptions.getBaseUrl().length() == 0
                        ? JuspayEnvironment.getBaseUrl()
                        : requestOptions.getBaseUrl();
            String endpoint = baseUrl + path;
            log.info("Executing request: " + method + " " + endpoint);
            log.info("Request parameters: ");
            // Printing this map separately to allow CardNumber and CVV filtering.
            log.info(mapMessage);

            if(method == RequestMethod.GET) {
                String serializedParams = serializeParams(params);
                if (serializedParams != null && !serializedParams.equals("")) {
                    endpoint = endpoint + "?" + serializedParams;
                }
            }

            URL url = new URL(endpoint);
            con = (HttpsURLConnection) url.openConnection();
            String sslProtocol = JuspayEnvironment.getSSLProtocol();
            SSLContext sslContext = SSLContext.getInstance(sslProtocol);
            sslContext.init(null, null, new SecureRandom());
            con.setSSLSocketFactory(sslContext.getSocketFactory());
            con.setConnectTimeout(requestOptions.getConnectTimeoutInMilliSeconds());
            con.setReadTimeout(requestOptions.getReadTimeoutInMilliSeconds());
            con.setRequestProperty("Content-Language", "en-US");
            con.setUseCaches (false);
            con.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
            con.setRequestProperty("User-Agent", JuspayEnvironment.SDK_VERSION);
            con.setRequestProperty("version", requestOptions.getApiVersion());
            con.setRequestProperty("x-merchantid", requestOptions.getMerchantId());

            Map<String, String> headers = requestOptions.getHeaders();
            if(headers != null) {
                for(String key : headers.keySet()) {
                    if(headers.get(key) != null) {
                        con.setRequestProperty(key, headers.get(key));
                    }
                }
            }

            switch (authMethod) {
                case JWE_JWS:
                    con.setRequestProperty("Content-Type", "application/json");
                    break;
                case BASIC:
                default:
                    String encodedKey = new String(Base64.encodeBase64(requestOptions.getApiKey().getBytes())).replaceAll("\n", "");
                    con.setRequestProperty("Authorization", String.format("Basic %s", encodedKey));
                    break;
            }

            if(method == RequestMethod.POST) {
                String payload = prepareParam(params, con.getRequestProperty("Content-Type"));
                con.setRequestMethod("POST");
                con.setRequestProperty("Content-Length", "" + Integer.toString(payload.getBytes().length));
                con.setDoInput(true);
                con.setDoOutput(true);
                DataOutputStream wr = new DataOutputStream(con.getOutputStream());
                wr.writeBytes(payload);
                wr.flush();
                wr.close();
            }
            inputStream = con.getInputStream();
            httpResponseCode = con.getResponseCode();
            reader = new BufferedReader(new InputStreamReader(inputStream, "UTF-8"));
            String line;
            StringBuilder response = new StringBuilder();
            while ((line = reader.readLine()) != null) {
                response.append(line);
            }
            responseString = response.toString();
            log.info("Received HTTP Response Code: " + httpResponseCode);
            log.info("Received response: " + responseString);
        } catch (SocketTimeoutException se) {
            log.error("Socket Timeout during request execution: ", se);
            throw new APIConnectionException(-1, "read_timeout", "read_timeout", se.getMessage());
        } catch (IOException ioe) {
            inputStream = con.getErrorStream();
            log.error(String.format("IOException while requesting %s : ", path), ioe);
            try {
                if(inputStream != null) {
                    httpResponseCode = con.getResponseCode();
                    String line;
                    reader = new BufferedReader(new InputStreamReader(inputStream));
                    StringBuilder response = new StringBuilder();
                    while ((line = reader.readLine()) != null) {
                        response.append(line);
                    }
                    responseString = response.toString();
                } else {
                    log.error("IOException occurred during request execution. Exception is: ", ioe);
                    throw new APIConnectionException(-1, "connection_error", "connection_error", ioe.getMessage());
                }
                log.info("Received HTTP Response Code: " + httpResponseCode);
                log.info("Received response: " + responseString);
            } catch (Exception e) {
                log.error("Exception occured during request execution. Exception is: ", e);
                throw new APIConnectionException(-1, "connection_error", "connection_error", e.getMessage());
            }
        } catch (InvalidKeysException e) {
            log.error("JWE JWS encryption keys are not configured:- " + e.getMessage());
            throw new InvalidRequestException(e.getHttpResponseCode(), e.getStatus(), e.getErrorCode(), e.getErrorMessage());
        } catch (Exception e) {
            log.error("Exception occurred during request execution. Exception is: ", e);
            throw new APIConnectionException(-1, "connection_error", "connection_error", e.getMessage());
        } finally {
            try {
                if(reader != null) {
                    reader.close();
                }
                if(inputStream != null) {
                    inputStream.close();
                }
                if(con != null) {
                    con.disconnect();
                }
            } catch (Exception e) {
                log.error("Exception occurred while closing the resources. Exception is: ", e);
                throw new APIConnectionException(-1, "connection_error", "connection_error", e.getMessage());
            }
        }
        JsonObject resJson = null;
        try {
            resJson = responseString != null ? new JsonParser().parse(responseString).getAsJsonObject() : null;
        } catch (JsonSyntaxException e) {
            // Do nothing, resJson will remain null.
            log.info("Not able to parse the response into Json. Exception is: ", e);
        }
        if (httpResponseCode >= 200 && httpResponseCode < 300) {
            switch (authMethod) {
                case JWE_JWS:
                    JsonObject decryptedResponse = decryptResponse(resJson, requestOptions.getJweJwsEncryptionKeys());
                    return decryptedResponse;
                case BASIC:
                default:
                    return resJson;
            }
        } else {
            String status = null;
            String errorCode = null;
            String errorMessage = null;
            if (resJson != null) {
                if (resJson.has("error") && !resJson.get("error").isJsonNull()) {
                    errorCode = status = resJson.get("error").getAsString();
                }
                if (resJson.has("status") && !resJson.get("status").isJsonNull()) {
                    status = resJson.get("status").getAsString();
                }
                if (resJson.has("error_code") && !resJson.get("error_code").isJsonNull()) {
                    errorCode = resJson.get("error_code").getAsString();
                }
                if (resJson.has("error_message") && !resJson.get("error_message").isJsonNull()) {
                    errorMessage = resJson.get("error_message").getAsString();
                }
            }
            switch (httpResponseCode) {
                case 400:
                case 404:
                    throw new InvalidRequestException(httpResponseCode, status, errorCode, errorMessage);
                case 401:
                    throw new AuthenticationException(httpResponseCode, status, errorCode, errorMessage);
                case 403:
                    throw new AuthorizationException(httpResponseCode, status, errorCode, errorMessage);
                default:
                    throw new APIException(httpResponseCode, "internal_error", "internal_error", "Something went wrong.");
            }
        }

    }

    protected static JsonObject decryptResponse(JsonObject data, JweJwsEncryptionKeys jweJwsEncryptionKeys)
            throws APIException {
        try {
            PrivateKey jwtPrivateKey = jweJwsEncryptionKeys.getJwsPrivateKey();
            PublicKey jwePublicKey = jweJwsEncryptionKeys.getJwePublicKey();

            String header = data.get("header").getAsString(),
                    encryptedKey = data.get("encryptedKey").getAsString(),
                    iv = data.get("iv").getAsString(),
                    encryptedPayload = data.get("encryptedPayload").getAsString(),
                    tag = data.get("tag").getAsString();

            String jweString = header + "." + encryptedKey + "." + iv + "." + encryptedPayload + "." + tag;

            String jweData = JWE.decrypt(jweString, jwtPrivateKey);
            TypeToken<Map<String, String>> typeToken = new TypeToken<Map<String, String>>() {};
            Map <String,String> jweMap = new Gson().fromJson(jweData, typeToken.getType());
            String jwsString = jweMap.get("header") + '.' + jweMap.get("payload") + '.' +  jweMap.get("signature");
            String finalResponse = JWS.verify(jwsString, jwePublicKey);

            return (finalResponse != null) ? new JsonParser().parse(finalResponse).getAsJsonObject() : null;

        } catch (JsonSyntaxException e) {
            log.info("Error occurred while parsing json, here's the log for error", e);
            throw new APIException(-1, "json_parsing_error", "json_parsing_error", e.getMessage());
        } catch (JuspayCryptoException e) {
            log.info("Error occurred while verifying the payload, here's the log for error", e);
            throw new APIException(-1, "cannot_get_response", "cannot_get_response", e.getMessage());
        } catch (Exception e) {
            log.info("Unknown error occurred while decryptResponse", e);
            throw new APIException(-1, "decrypt_request_error", "decrypt_request_error", e.getMessage());
        }
    }

    protected static Map<String, Object> encryptRequest(Map<String, Object> params, JweJwsEncryptionKeys jweJwsEncryptionKeys)
            throws APIException  {
        try {
            Gson gson = new Gson();
            String data = gson.toJson(params), keyId = jweJwsEncryptionKeys.getKeyId();
            PrivateKey jwtPrivateKey = jweJwsEncryptionKeys.getJwsPrivateKey();
            PublicKey jwePublicKey = jweJwsEncryptionKeys.getJwePublicKey();

            String signedJWS = JWS.sign(data, keyId, jwtPrivateKey);

            String[] signedPayloadParts = signedJWS.split("\\.");
            Map <String,String> signedJSON = new HashMap <String,String>();
            signedJSON.put("signature",signedPayloadParts[2]);
            signedJSON.put("payload",signedPayloadParts[1]);
            signedJSON.put("header",signedPayloadParts[0]);

            String json = gson.toJson(signedJSON);

            String jweString = JWE.encrypt(json, keyId, jwePublicKey);

            String[] encryptedPayloadParts = jweString.split("\\.");
            Map <String, Object> reqJSON = new HashMap <String,Object>();
            reqJSON.put("header",encryptedPayloadParts[0]);
            reqJSON.put("encryptedKey",encryptedPayloadParts[1]);
            reqJSON.put("iv",encryptedPayloadParts[2]);
            reqJSON.put("encryptedPayload",encryptedPayloadParts[3]);
            reqJSON.put("tag",encryptedPayloadParts[4]);

            return reqJSON;
        } catch (JuspayCryptoException e) {
            log.info("Error occurred while signing the payload, here's the log for error", e);
            throw new APIException(-1, "cannot_prepare_payload", "cannot_prepare_payload", e.getMessage());
        } catch (Exception e) {
            log.info("Unknown error occurred while encryptRequest", e);
            throw new APIException(-1, "encrypt_request_error", "encrypt_request_error", e.getMessage());
        }
    }

    protected static JsonObject addInputParamsToResponse(Map<String, Object> params, JsonObject response) {
        JsonObject inputJson = new JsonParser().parse(new GsonBuilder().create().toJson(params)).getAsJsonObject();
        Iterator<Map.Entry<String, JsonElement>> iterator = inputJson.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, JsonElement> entry = iterator.next();
            response.add(entry.getKey(), entry.getValue());
        }
        return response;
    }

    protected static <T> T createEntityFromResponse(JsonElement response, Class<T> entityClass) {
        Gson gson = new GsonBuilder()
                .setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES)
                .create();
        T entity = gson.fromJson(response, entityClass);
        return entity;
    }

    protected enum RequestMethod {
        GET, POST, DELETE
    }

    protected enum AuthMethod {
        BASIC, JWE_JWS
    }

}