package in.juspay.model;

import java.util.LinkedHashMap;
import java.util.Map;

public class RequestOptions {
    private String apiKey;
    private String merchantId;
    private String apiVersion;
    private int connectTimeoutInMilliSeconds;
    private int readTimeoutInMilliSeconds;
    private JweJwsEncryptionKeys jweJwsEncryptionKeys;
    private Map<String, String> headers;
    private String baseUrl;

    private RequestOptions() {
        apiKey = JuspayEnvironment.getApiKey();
        merchantId = JuspayEnvironment.getMerchantId();
        connectTimeoutInMilliSeconds = JuspayEnvironment.getConnectTimeoutInMilliSeconds();
        readTimeoutInMilliSeconds = JuspayEnvironment.getReadTimeoutInMilliSeconds();
        apiVersion = JuspayEnvironment.API_VERSION;
        jweJwsEncryptionKeys = JuspayEnvironment.getJweJwsEncryptionKeys();
        headers = new LinkedHashMap<String, String>();
        baseUrl = JuspayEnvironment.getBaseUrl();
    }

    public static RequestOptions createDefault() {
        return new RequestOptions();
    }

    public String getApiKey() {
        return apiKey;
    }

    public String getMerchantId() {
        return merchantId;
    }

    public RequestOptions withMerchantId(String merchantId) {
        this.merchantId = merchantId;
        return this;
    }

    public RequestOptions withApiKey(String apiKey) {
        this.apiKey = apiKey;
        return this;
    }

    public int getConnectTimeoutInMilliSeconds() {
        return connectTimeoutInMilliSeconds;
    }

    public RequestOptions withConnectTimeout(int connectTimeout) {
        this.connectTimeoutInMilliSeconds = connectTimeout;
        return this;
    }

    public int getReadTimeoutInMilliSeconds() {
        return readTimeoutInMilliSeconds;
    }

    public RequestOptions withReadTimeout(int readTimeout) {
        this.readTimeoutInMilliSeconds = readTimeout;
        return this;
    }

    public String getApiVersion() {
        return apiVersion;
    }

    public RequestOptions withApiVersion(String version) {
        this.apiVersion = version;
        return this;
    }

    public RequestOptions withJweJwsEncryption(JweJwsEncryptionKeys encryption) {
        this.jweJwsEncryptionKeys = encryption;
        return this;
    }

    public JweJwsEncryptionKeys getJweJwsEncryptionKeys() {
        return jweJwsEncryptionKeys;
    }

    public Map<String, String> getHeaders() {
        return headers;
    }

    public String getHeaders(String key) {
        return headers.get(key);
    }

    public RequestOptions withHeaders(Map<String, String> val) {
        if(val != null) {
            this.headers = val;
        }
        return this;
    }

    public RequestOptions withHeaders(String key, String val) {
        if(key != null && val != null && val.length() != 0 && key.length() != 0) {
            this.headers.put(key, val);
        }
        return this;
    }

    public RequestOptions withCustomerId(String customerId) {
        this.withHeaders("x-customerid", customerId);
        return this;
    }

    public RequestOptions withRoutingId(String routingId) {
        this.withHeaders("x-routing-id", routingId);
        return this;
    }

    public String getRoutingId() {
        return headers.get("x-routing-id");
    }

    public String getBaseUrl() {
        return baseUrl;
    }

    public RequestOptions withBaseUrl(String baseUrl) {
        this.baseUrl = baseUrl;
        return this;
    }
}
