package `in`.navanatech.zabaan.publishersdk

import `in`.navanatech.zabaan.publishersdk.lifecycle.ApplicationLifecycleObserver
import `in`.navanatech.zabaan.publishersdk.lifecycle.ScreenTracker
import `in`.navanatech.zabaan.publishersdk.providers.RepositoryProvider
import `in`.navanatech.zabaan.publishersdk.providers.UiProvider
import `in`.navanatech.zabaan.publishersdk.util.Logger
import `in`.navanatech.zabaan.publishersdk.util.NotificationHelper
import `in`.navanatech.zabaan.publishersdk.util.NotificationType
import `in`.navanatech.zabaan.publishersdk.util.e
import android.app.Application
import android.content.IntentFilter
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.ProcessLifecycleOwner
import androidx.lifecycle.lifecycleScope
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach

class ZabaanPublisher private constructor(private val application: Application) {

    private val screenTracker = ScreenTracker()
    private val applicationLifecycleObserver = ApplicationLifecycleObserver()
    private val bubbleManager = UiProvider.provideBubbleManager()
    private val repository = RepositoryProvider.provideRepository()
    private val captureManager = CaptureManager(repository)
    private val bubbleBroadcastReceiver by lazy { EnableBubbleActionBroadcastReceiver() }

    init {
        initializeLibrary()
    }

    private fun initializeLibrary() {
        // Check if Zabaan SDK is available
        if (!isZabaanAvailable()) {
            NotificationHelper.showNotification(application, NotificationType.ERROR)
            return
        }
        // Start tracking application events
        ProcessLifecycleOwner.get().lifecycle.addObserver(applicationLifecycleObserver)
        // Start tracking activity and fragment changes when the class is created.
        application.registerActivityLifecycleCallbacks(screenTracker)
        setupObservers()
    }

    private fun setupObservers() {
        applicationLifecycleObserver.applicationEventFlow.onEach { event ->
            when (event) {
                Lifecycle.Event.ON_RESUME -> {
                    application.registerReceiver(
                        bubbleBroadcastReceiver,
                        IntentFilter("in.navanatech.zabaan.publisher.ENABLE_BUBBLE")
                    )
                    NotificationHelper.showNotification(
                        this.application,
                        if (repository.isTokenAvailable.value) NotificationType.CONNECTED else NotificationType.START
                    )
                }
                Lifecycle.Event.ON_PAUSE -> {
                    NotificationHelper.cancelNotification(this.application)
                    application.unregisterReceiver(bubbleBroadcastReceiver)
                }
                Lifecycle.Event.ON_DESTROY -> {
                    NotificationHelper.cancelNotification(this.application)
                }
                else -> {
                }
            }
        }.launchIn(ProcessLifecycleOwner.get().lifecycleScope)

        repository.isTokenAvailable.onEach { success ->
            e { "repository token available" }
            NotificationHelper.showNotification(this.application, NotificationType.CONNECTED)
            if (success)
                screenTracker.activityFlow.onEach { activity ->
                    if (activity != null)
                        bubbleManager.show(activity) { captureManager.captureScreen(activity) }
                    else
                        bubbleManager.clear()
                }.launchIn(ProcessLifecycleOwner.get().lifecycleScope)
        }.launchIn(ProcessLifecycleOwner.get().lifecycleScope)
    }

    private fun isZabaanAvailable(): Boolean {
        return try {
            Class.forName("com.zabaan.sdk.Zabaan")
            true
        } catch (e: ClassNotFoundException) {
            false
        }
    }

    private fun flushClass() {
        Logger.error("Inside flush", null)
        screenTracker.clearActivityFlow()
        application.unregisterActivityLifecycleCallbacks(screenTracker)
        ProcessLifecycleOwner.get().lifecycle.removeObserver(applicationLifecycleObserver)
    }

    companion object {
        private lateinit var INSTANCE: ZabaanPublisher

        @JvmStatic
        fun init(application: Application, enableLogs: Boolean = false) {
            Logger.init(enableLogs)
            INSTANCE = ZabaanPublisher(application)
        }

        @JvmStatic
        fun flush() {
            if (::INSTANCE.isInitialized)
                INSTANCE.flushClass()
        }
    }
}
