package `in`.navanatech.zabaan.publishersdk.data

import `in`.navanatech.zabaan.publishersdk.data.network.PublisherService
import `in`.navanatech.zabaan.publishersdk.model.ViewNode
import `in`.navanatech.zabaan.publishersdk.model.ViewNodeJsonAdapter
import `in`.navanatech.zabaan.publishersdk.util.e
import `in`.navanatech.zabaan.publishersdk.util.v
import com.skydoves.sandwich.*
import com.squareup.moshi.Moshi
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.withContext
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.MultipartBody
import okhttp3.RequestBody.Companion.asRequestBody
import java.io.File

/**
 * Publisher repository
 *
 * @property publisher
 * @property dispatcher
 * @constructor Create empty Publisher repository
 */
class PublisherRepository(
    private val publisher: PublisherService,
    private val dispatcher: CoroutineDispatcher = Dispatchers.IO
) {

    private var token = ""

    private val _isTokenAvailable = MutableStateFlow(false)
    val isTokenAvailable = _isTokenAvailable.asStateFlow()

    suspend fun getToken(code: String, onComplete: suspend (Boolean) -> Unit) =
        withContext(dispatcher) {
            publisher.getToken(code).suspendOnSuccess {
                token = ("Bearer " + data.accessToken)
                _isTokenAvailable.value = true
                onComplete(true)
            }.suspendOnError {
                e { errorBody.toString() }
                _isTokenAvailable.value = false
                onComplete(false)
            }.suspendOnException {
                e { message() }
                _isTokenAvailable.value = false
                onComplete(false)
            }
        }

    suspend fun publishScreenshot(viewNode: ViewNode, screenshotName: String, file: File) {
        val imagePart: MultipartBody.Part = MultipartBody.Part.createFormData(
            "screenshot",
            "$screenshotName.jpeg",
            file.asRequestBody("image/jpeg".toMediaType())
        )
        val multipartBody = MultipartBody.Part.createFormData(
            "metadata_json",
            ViewNodeJsonAdapter(Moshi.Builder().build()).toJson(viewNode) // TODO: Cleanup
        )
        val screenshotInfo = MultipartBody.Part.createFormData(
            "identifier",
            screenshotName
        )
        val versionInfo = MultipartBody.Part.createFormData(
            "version",
            "1"
        )
        e {
            ViewNodeJsonAdapter(
                Moshi.Builder().build()
            ).toJson(viewNode)
        }
        publisher.screenDetails(token, multipartBody, imagePart, screenshotInfo, versionInfo)
            .suspendOnSuccess {
                v { data.msg }
            }.suspendOnError {
                e { errorBody?.string() ?: "Error" }
            }.suspendOnException {
                e { message() }
            }.suspendOnFailure {
                e { this }
            }
    }
}