package `in`.navanatech.zabaan.publishersdk.parsing

import `in`.navanatech.zabaan.publishersdk.model.ViewNode
import `in`.navanatech.zabaan.publishersdk.util.className
import `in`.navanatech.zabaan.publishersdk.util.e
import `in`.navanatech.zabaan.publishersdk.util.stringId
import android.view.View
import android.view.ViewGroup
import android.widget.ScrollView
import androidx.core.view.ScrollingView
import androidx.core.view.children
import androidx.core.widget.NestedScrollView
import androidx.recyclerview.widget.RecyclerView
import androidx.viewpager2.widget.ViewPager2

/**
 * View tree parser
 * A class to recursively parse entire view tree from a given view
 *
 * @constructor Create empty View tree parser
 */
object ViewTreeParser {

    fun parseViewTree(rootView: View): ViewNode {
        return recursiveParser(rootView, null)
            ?: error("Cannot parse the view tree, returned node is null")
    }

    private fun recursiveParser(view: View, parent: View?): ViewNode? {
        val x = view.x
        var y = view.y
        val intId = view.id
        val stringId = view.stringId.split("/").last()
        val children = mutableListOf<ViewNode>()
        var height = view.height
        val width = view.width
        val isInteractive = view.isClickable
        val isScrollable = view.isScrollable()
        val viewName = view.className

        if (stringId == "zzz_bubble_overlay") {
            return null
        }

        val adapterPosition =
            if (parent is RecyclerView && (parent.parent !is ViewPager2 && parent.className != "RecyclerViewImpl")) parent.getChildAdapterPosition(
                view
            ) else -1

        if (parent is NestedScrollView || parent is ScrollView) {
//            y -= parent.scrollY
            height = parent.height
        }

        parent?.parent?.takeIf { it is NestedScrollView || it is ScrollView }?.let {
            if (it is NestedScrollView)
                y -= it.scrollY
            if (it is ScrollView)
                y -= it.scrollY
        }

        if (view is ViewGroup) {
            view.children.forEach { childView ->
                val childViewNode = recursiveParser(childView, view)
                if (childViewNode != null) children.add(childViewNode)
            }
        }

        return ViewNode(
            intId,
            stringId,
            x,
            y,
            width,
            height,
            children,
            isInteractive,
            isScrollable,
            viewName,
            adapterPosition
        )
    }

    private const val SCROLL_UP = -1
    private const val SCROLL_DOWN = 1
    private const val SCROLL_LEFT = -1
    private const val SCROLL_RIGHT = 1

    private fun View.isScrollable() =
        canScrollVertically(SCROLL_UP) ||
                canScrollVertically(SCROLL_DOWN) ||
                canScrollHorizontally(SCROLL_LEFT) ||
                canScrollHorizontally(SCROLL_RIGHT)

}